<?php

namespace App\Http\Controllers\API\Administrations;

use App\Http\Controllers\Controller;
use App\Models\Branch;
use App\Models\Traits\HttpResponses;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Gate;
use Symfony\Component\HttpFoundation\Response;

class BranchApiController extends Controller
{
    use HttpResponses;

    public function index(Request $request)
    {
        if (!Gate::allows('View Branch')) {
            return $this->error('Permission Denied', [], Response::HTTP_FORBIDDEN);
        }

        $limit = $request->query->get('limit', 15);
        $search = $request->query->get('search');

        $query = Branch::latest('name');

        if (!empty($search)) {
            $query->where('name', 'LIKE', "%{$search}%");
        }

        $branches = $query->paginate($limit);

        $data = [];
        foreach ($branches as $branch) {
            $data[] = [
                'id' => $branch->id,
                'name' => $branch->name,
                'added_date' => Carbon::parse($branch->created_at)->format('d:m:Y h:i')
            ];
        }

        return $this->success([
            'branches' => $data,
            'pagination' => [
                'total' => $branches->total(),
                'current_page' => $branches->currentPage(),
                'last_page' => $branches->lastPage(),
                'per_page' => $branches->perPage()
            ]
        ], 'Data Retrieved.', true);
    }

    public function branchForSelect(Request $request)
    {
        $limit = $request->query('limit', 50);
        $search = trim($request->query('search'));

        $query = Branch::select('id', 'name')->orderBy('name');

        if (!empty($search)) {
            $query->where('name', 'LIKE', "%{$search}%");
        }

        $branches = $query->limit($limit)->get();

        $data = [];
        foreach ($branches as $branch) {
            $data[] = [
                'id' => $branch->id,
                'text' => $branch->name,
            ];
        }

        return $this->success(['branches' => $data], 'Data Retrieved.');
    }
}
