<?php

namespace App\Http\Controllers\API\Administrations;

use App\Http\Controllers\Controller;
use App\Models\Employee;
use App\Models\Traits\HttpResponses;
use Exception;
use Illuminate\Http\Request;
use Illuminate\Support\Carbon;
use Illuminate\Support\Facades\Gate;
use Illuminate\Support\Facades\Validator;
use PhpParser\Node\Expr\Empty_;
use Symfony\Component\HttpFoundation\Response;

class EmployeeApiController extends Controller
{
    use HttpResponses;

    public function index(Request $request)
    {
        if (!Gate::allows('View Employee')) {
            return $this->error('Permission Denied', [], Response::HTTP_FORBIDDEN);
        }

        $limit = $request->query->get('limit', 15);
        $search = $request->query->get('search');

        $query = Employee::orderBy('name')->where('status',1);

        if (!empty($search)) {
            $query->where('name', 'LIKE', "%{$search}%")
                ->orWhere('phone', 'LIKE', "%{$search}%")
                ->orWhere('email', 'LIKE', "%{$search}%");
        }

        $employees = $query->paginate($limit);

        $data = [];
        foreach ($employees as $employee) {
            $data[] = [
                'id' => $employee->id,
                'name' => $employee->name,
                'email' => $employee->email ?? '',
                'phone' => $employee->phone ?? '',
                'designation' => $employee->designation ?? '',
                'department' => $employee->department ?? '',
                'branch' => $employee->branch ?? '',
                'status' => $employee->employee_status,
                'added_date' => Carbon::parse($employee->created_at)->format('d:m:Y h:i'),
                'profile_picture' => $employee->profile_picture ?? 'user.png'
            ];
        }

        return $this->success([
            'employees' => $data,
            'pagination' => [
                'total' => $employees->total(),
                'current_page' => $employees->currentPage(),
                'last_page' => $employees->lastPage(),
                'per_page' => $employees->perPage()
            ]
        ], 'Data Retrieved.', true);
    }

    public function employees(Request $request)
    {
        if (!Gate::allows('View Employee')) {
            return $this->error('Permission Denied', [], Response::HTTP_FORBIDDEN);
        }

        $limit = $request->query->get('limit', 15);
        $search = $request->query->get('search');

        $query = Employee::orderBy('name');

        if (!empty($search)) {
            $query->where('name', 'LIKE', "%{$search}%")
                ->orWhere('phone', 'LIKE', "%{$search}%")
                ->orWhere('email', 'LIKE', "%{$search}%");
        }

        $employees = $query->paginate($limit);

        $data = [];
        foreach ($employees as $employee) {
            $data[] = [
                'id' => $employee->id,
                'name' => $employee->name,
                'email' => $employee->email ?? '',
                'phone' => $employee->phone ?? '',
                'designation' => $employee->designation ?? '',
                'department' => $employee->department ?? '',
                'branch' => $employee->branch ?? '',
                'status' => $employee->employee_status,
                'added_date' => Carbon::parse($employee->created_at)->format('d:m:Y h:i'),
                'profile_picture' => $employee->profile_picture ?? 'user.png'
            ];
        }

        return $this->success([
            'employees' => $data,
            'pagination' => [
                'total' => $employees->total(),
                'current_page' => $employees->currentPage(),
                'last_page' => $employees->lastPage(),
                'per_page' => $employees->perPage()
            ]
        ], 'Data Retrieved.', true);
    }

    public function employeeForSelect(Request $request)
    {
        $limit = $request->query('limit', 50);
        $search = trim($request->query('search'));

        $query = Employee::select('id', 'name')->orderBy('name');

        if (!empty($search)) {
            $query->where('name', 'LIKE', "%{$search}%");
        }

        $employees = $query->limit($limit)->get();

        $data = [];
        foreach ($employees as $employee) {
            $data[] = [
                'id' => $employee->id,
                'text' => ucfirst($employee->name),
            ];
        }

        return $this->success(['employees' => $data], 'Data Retrieved.');
    }

    public function show(Request $request)
    {
        if (!Gate::allows('View Employee')) {
            return $this->error('Permission Denied', [], Response::HTTP_FORBIDDEN);
        }

        $validate = Validator::make($request->all(), [
            'employee_id' => 'required|exists:employees,id'
        ]);

        if ($validate->fails()) {
            return $this->error('Validation Failed.', $validate->errors(), Response::HTTP_UNPROCESSABLE_ENTITY);
        }

        try {
            $employee = Employee::find($request->employee_id);

            if (!$employee) {
                return $this->error('Employee not found.', [], Response::HTTP_NOT_FOUND);
            }

            $image = $employee->profile_picture ?? 'user.png';
            $url = config('app.hr_url');

            $data = [
                'id' => $employee->id,
                'name' => $employee->name,
                'email' => $employee->email ?? '',
                'phone' => $employee->phone ?? '',
                'status' => $employee->employee_status ?? '',
                'designation' => $employee->designation ?? '',
                'department' => $employee->department ?? '',
                'branch' => $employee->branch ?? '',
                'profile_picture' => $url.'/public/images/'.$image,
            ];

            $current_allocations = $employee->assetAllocations
                ->where('status', 1)
                ->map(function ($allocation) {
                    return [
                        'asset' => $allocation->asset->asset_name ?? '',
                        'serial_number' => $allocation->asset->serial_number ?? '',
                        'branch' => $allocation->branch->name ?? '',
                        'assigned_date' => $allocation->date_assigned,
                        'assigned_by' => $allocation->allocatedBy->name ?? '',
                    ];
                })
                ->values(); // r

            $allocation_history = $employee->assetAllocations
                ->where('status', 0)
                ->map(function ($allocation) {
                    return [
                        'asset' => $allocation->asset->asset_name ?? '',
                        'serial_number' => $allocation->asset->serial_number ?? '',
                        'branch' => $allocation->branch->name ?? '',
                        'assigned_date' => $allocation->date_assigned,
                        'date_unassigned' => $allocation->date_unassigned ?? '',
                        'assigned_by' => $allocation->allocatedBy->name ?? '',
                        'unallocated_by' => $allocation->unallocatedBy->name ?? '',
                        'reasons' => $allocation->reasons ?? '',
                    ];
                })
                ->values();

            return $this->success([
                'employee' => $data,
                'current_allocations' => $current_allocations->isNotEmpty() ? $current_allocations : [],
                'allocation_history' => $allocation_history,
            ], 'Data Retrieved.');
        } catch (Exception $e) {
            return $this->error('Error occurred.', $e->getMessage(), Response::HTTP_UNPROCESSABLE_ENTITY);
        }
    }
}
