<?php

namespace App\Http\Controllers\API\ApplicationAssets;

use App\Http\Controllers\Controller;
use App\Models\ApplicationAsset;
use App\Models\ApplicationAssetVendor;
use App\Models\ApplicationAssetVendorContract;
use App\Models\Traits\HttpResponses;
use Exception;
use Illuminate\Http\Request;
use Illuminate\Http\UploadedFile;
use Illuminate\Support\Carbon;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Gate;
use Illuminate\Support\Facades\Validator;
use Symfony\Component\HttpFoundation\Response;

class ApplicationAssetApiController extends Controller
{
    use HttpResponses;

    public function index(Request $request)
    {
        if (!Gate::allows('View Software Asset') || !Gate::allows('View Service Asset')) {
            return $this->error('Permission Denied', [], Response::HTTP_FORBIDDEN);
        }

        $validate = Validator::make($request->all(), [
            'type' => 'required|in:SOFTWARE,SERVICE',
        ]);

        if ($validate->fails()) {
            return $this->error('Validation Failed.', $validate->errors(), Response::HTTP_UNPROCESSABLE_ENTITY);
        }

        $limit = $request->query('limit', 15);
        $search = $request->query('search');
        $type = $request->type;

        $query = ApplicationAsset::where('type', $type)
            ->latest();

        if (!empty($search)) {
            $query->where('name', 'LIKE', "%{$search}%");
        }

        $application_assets = $query->paginate($limit);

        $data = [];
        foreach ($application_assets as $application_asset) {
            $data[] = [
                'id' => $application_asset->id,
                'name' => ucwords($application_asset->name),
                'ownership' => $application_asset->ownership->name,
                'requisition_date' => $application_asset->requisition_date,
                'createdBy' => $application_asset->createdBy->name,
                'added_date' => Carbon::parse($application_asset->created_at)->format('d:m:Y h:i')
            ];
        }

        return $this->success([
            'application_assets' => $data,
            'pagination' => [
                'total' => $application_assets->total(),
                'current_page' => $application_assets->currentPage(),
                'last_page' => $application_assets->lastPage(),
                'per_page' => $application_assets->perPage()
            ]
        ], 'Data Retrieved.', true);
    }

    public function store(Request $request)
    {
        if (!Gate::allows('Create Software Asset') && !Gate::allows('Create Service Asset')) {
            return $this->error('Permission Denied', [], Response::HTTP_FORBIDDEN);
        }

        $validate = Validator::make($request->all(), [
            'name' => 'required|string|unique:application_assets,name',
            'type' => 'required|in:SOFTWARE,SERVICE',
            'ownership_id' => 'nullable|exists:vendors,id',
            'requisition_date' => 'required|date',
            'descriptions' => 'nullable|string',

            'item' => 'required|array|min:1',
            'item.*.vendor_ids' => 'required|exists:vendors,id',
            'item.*.branch_ids' => 'nullable|exists:branches,id',
            'item.*.start_dates' => 'required|date',
            'item.*.end_dates' => 'required|date|after_or_equal:item.*.start_dates',
            'item.*.document_path' => 'nullable|file|mimes:pdf|max:5120',
        ]);

        if ($validate->fails()) {
            return $this->error('Validation Failed.', $validate->errors(), Response::HTTP_UNPROCESSABLE_ENTITY);
        }

        try {
            DB::beginTransaction();

            $application_asset = ApplicationAsset::create([
                'ownership_id' => $request->ownership_id,
                'name' => $request->name,
                'type' => $request->type,
                'requisition_date' => $request->requisition_date,
                'descriptions' => $request->descriptions,
                'created_by' => Auth::id(),
                'status' => 1,
            ]);

            foreach ($request->item as $data) {
                $path = null;

                if (isset($data['document_path']) && $data['document_path'] instanceof UploadedFile) {
                    $path = $data['document_path']->store('contracts', 'public');
                }

                $asset_vendor = ApplicationAssetVendor::create([
                    'application_asset_id' => $application_asset->id,
                    'vendor_id' => $data['vendor_ids'],
                    'branch_id' => $data['branch_ids'] ?? null,
                    'date' => Carbon::today(),
                    'created_by' => Auth::id(),
                ]);

                ApplicationAssetVendorContract::create([
                    'application_asset_vendor_id' => $asset_vendor->id,
                    'start_date' => $data['start_dates'],
                    'end_date' => $data['end_dates'],
                    'document_path' => $path,
                    'created_by' => Auth::id(),
                    'status' => 1,
                ]);
            }

            DB::commit();
            return $this->success([], 'Record Created.', Response::HTTP_CREATED, true);
        } catch (Exception $e) {
            DB::rollBack();
            return $this->error('Error Occurred.', $e->getMessage(), Response::HTTP_UNPROCESSABLE_ENTITY);
        }
    }

    public function details(Request $request)
    {
        if (!Gate::allows('View Software Asset') && !Gate::allows('View Service Asset')) {
            return $this->error('Permission Denied', [], Response::HTTP_FORBIDDEN);
        }

        $validate = Validator::make($request->all(), [
            'application_asset_id' => 'required|exists:application_assets,id',
        ]);

        if ($validate->fails()) {
            return $this->error('Validation Failed.', $validate->errors(), Response::HTTP_UNPROCESSABLE_ENTITY);
        }

        try {
            $application_asset = ApplicationAsset::find($request->application_asset_id);

            $data = [
                'id' => $application_asset->id,
                'name' => $application_asset->name,
                'type' => $application_asset->type,
                'ownership' => $application_asset->ownership->name ?? '',
                'requisition_date' => $application_asset->requisition_date,
                'status' => $application_asset->status ? 'Active' : 'Inactive',
                'descriptions' => $application_asset->descriptions ?? '',
            ];

            $current_contracts = [];
            $previous_contracts = [];
            $upcoming_contracts = [];

            foreach ($application_asset->applicationAssetVendors as $application_asset_vendor) {
                foreach ($application_asset_vendor->applicationAssetVendorContracts as $application_asset_vendor_contract) {

                    $start = Carbon::parse($application_asset_vendor_contract->start_date);
                    $end = Carbon::parse($application_asset_vendor_contract->end_date);
                    $now = now();

                    $contract_data = [
                        'vendor' => $application_asset_vendor->vendor->name ?? '',
                        'branch' => $application_asset_vendor->branch->name ?? '',
                        'start_date' => $application_asset_vendor_contract->start_date,
                        'end_date' => $application_asset_vendor_contract->end_date,
                        'document_path' => $application_asset_vendor_contract->document_path
                            ? asset('storage/' . $application_asset_vendor_contract->document_path)
                            : '',
                        'created_by' => $application_asset_vendor_contract->createdBy->name ?? '',
                        'renewals' => $application_asset_vendor_contract->applicationAssetVendorContractRenewals->map(function ($renewal) {
                            return [
                                'start_date' => $renewal->start_date,
                                'end_date' => $renewal->end_date,
                                'cost' => $renewal->cost,
                                'descriptions' => $renewal->descriptions,
                                'created_by' => $renewal->createdBy->name ?? '',
                            ];
                        })->values(),
                    ];

                    if ($now->between($start, $end)) {
                        $current_contracts[] = $contract_data;
                    } elseif ($start->isFuture()) {
                        $upcoming_contracts[] = $contract_data;
                    } else {
                        $previous_contracts[] = $contract_data;
                    }
                }
            }

            return $this->success([
                'application_asset' => $data,
                'current_contracts' => $current_contracts,
                'upcoming_contracts' => $upcoming_contracts,
                'previous_contracts' => $previous_contracts,
            ], 'Data Retrieved.');
        } catch (Exception $e) {
            return $this->error('Error Occurred.', $e->getMessage(), Response::HTTP_UNPROCESSABLE_ENTITY);
        }
    }
}
