<?php

namespace App\Http\Controllers\API\ApplicationAssets;

use App\Http\Controllers\Controller;
use App\Models\ApplicationAsset;
use App\Models\ApplicationAssetVendor;
use App\Models\ApplicationAssetVendorContract;
use App\Models\Traits\HttpResponses;
use Exception;
use Illuminate\Http\Request;
use Illuminate\Support\Carbon;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Gate;
use Illuminate\Support\Facades\Validator;
use Symfony\Component\HttpFoundation\Response;

class ApplicationAssetVendorApiController extends Controller
{
    use HttpResponses;
    public function store(Request $request)
    {
        if (!Gate::allows('Create Software Asset Vendor') && !Gate::allows('Create Service Asset Vendor')) {
            return $this->error('Permission Denied', [], Response::HTTP_FORBIDDEN);
        }

        $validate = Validator::make($request->all(), [
            'application_asset_id' => 'required|exists:application_assets,id',
            'vendor_id' => 'required|exists:vendors,id',
            'branch_id' => 'nullable|exists:branches,id',
            'start_date' => 'required|date',
            'end_date' => 'required|date|after:start_date',
            'document_path' => 'nullable|file|mimes:pdf|max:5120',
        ]);

        if ($validate->fails()) {
            return $this->error('Validation Failed.', $validate->errors(), Response::HTTP_UNPROCESSABLE_ENTITY);
        }

        try {
            DB::beginTransaction();

            $application_asset = ApplicationAsset::findOrFail($request->application_asset_id);

            $exists = ApplicationAssetVendor::where('application_asset_id', $application_asset->id)
                ->where('vendor_id', $request->vendor_id)
                ->exists();

            if ($exists) {
                return $this->error('Vendor already assigned to this application asset.', [], Response::HTTP_CONFLICT);
            }

            $document_path = null;
            if ($request->hasFile('document_path')) {
                $document_path = $request->file('document_path')->store('contracts', 'public');
            }

            $asset_vendor = ApplicationAssetVendor::create([
                'application_asset_id' => $application_asset->id,
                'vendor_id' => $request->vendor_id,
                'branch_id' => $request->branch_id,
                'date' => Carbon::today(),
                'created_by' => Auth::id(),
            ]);

            ApplicationAssetVendorContract::create([
                'application_asset_vendor_id' => $asset_vendor->id,
                'start_date' => $request->start_date,
                'end_date' => $request->end_date,
                'document_path' => $document_path,
                'created_by' => Auth::id(),
                'status' => 1,
            ]);

            DB::commit();
            return $this->success([], 'Record created.', Response::HTTP_CREATED, true);
        } catch (Exception $e) {
            DB::rollBack();
            return $this->error('Error Occurred.', $e->getMessage(), Response::HTTP_UNPROCESSABLE_ENTITY);
        }
    }
}
