<?php

namespace App\Http\Controllers\API\ApplicationAssets;

use App\Http\Controllers\Controller;
use App\Models\ApplicationAssetVendor;
use App\Models\ApplicationAssetVendorContract;
use App\Models\Traits\HttpResponses;
use Carbon\Carbon;
use Exception;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Gate;
use Illuminate\Support\Facades\Validator;
use Symfony\Component\HttpFoundation\Response;

class ApplicationAssetVendorContractApiController extends Controller
{
    use HttpResponses;

    public function index(Request $request)
    {
        if (!Gate::allows('View Software Asset Vendor Contract') && !Gate::allows('View Service Asset Vendor Contract')) {
            return $this->error('Permission Denied', [], 403);
        }

        $limit = $request->query('limit', 15);
        $search = $request->query('search');
        $vendor_ids = $request->vendor_ids;

        $query = ApplicationAssetVendorContract::with([
            'applicationAssetVendor.vendor',
            'applicationAssetVendor.branch',
            'applicationAssetVendor.applicationAsset',
        ])->latest();

        if (!empty($search)) {
            $query->whereHas('applicationAssetVendor.vendor', function ($query) use ($search) {
                $query->where('name', 'LIKE', "%{$search}%");
            })
                ->orWhereHas('applicationAssetVendor.applicationAsset', function ($query) use ($search) {
                    $query->where('name', 'LIKE', "%{$search}%");
                });
        }

        if (!empty($vendor_ids)) {
            $query->whereHas('applicationAssetVendor', function ($query) use ($vendor_ids) {
                $query->whereIn('vendor_id', $vendor_ids);
            });
        }

        $contracts = $query->paginate($limit);

        $data = [];
        foreach ($contracts as $contract) {
            $data[] =  [
                'id' => $contract->id,
                'vendor' => $contract->applicationAssetVendor->vendor->name ?? '',
                'application_asset_name' => $contract->applicationAssetVendor->applicationAsset->name ?? '',
                'branch' => $contract->applicationAssetVendor->branch->name ?? '',
                'start_date' => Carbon::parse($contract->start_date)->format('M d, Y'),
                'end_date' => Carbon::parse($contract->end_date)->format('M d, Y'),
                'status' => $contract->contract_status,
                'document_path' => $contract->document_path ?? '',
                'added_date' => Carbon::parse($contract->created_at)->format('d-m-Y H:i'),
            ];
        }

        return $this->success([
            'contracts' => $data,
            'pagination' => [
                'total' => $contracts->total(),
                'current_page' => $contracts->currentPage(),
                'last_page' => $contracts->lastPage(),
                'per_page' => $contracts->perPage()
            ]
        ], 'Data Retrieved.', true);
    }

    public function store(Request $request)
    {
        if (!Gate::allows('Create Software Asset Vendor Contract') && !Gate::allows('Create Service Asset Vendor Contract')) {
            return $this->error('Permission Denied', [], 403);
        }

        $validate = Validator::make($request->all(), [
            'application_asset_vendor_id' => 'required|exists:application_asset_vendors,id',
            'start_date' => 'required|date',
            'end_date' => 'required|date|after:start_date',
            'document_path' => 'nullable|file|mimes:pdf|max:2048',
        ]);

        if ($validate->fails()) {
            return $this->error('Validation Failed.', $validate->errors(), Response::HTTP_UNPROCESSABLE_ENTITY);
        }

        try {
            $application_asset_vendor = ApplicationAssetVendor::findOrFail($request->application_asset_vendor_id);

            $existing_active = ApplicationAssetVendorContract::where('application_asset_vendor_id', $application_asset_vendor->id)
                ->whereDate('start_date', '<=', now())
                ->whereDate('end_date', '>=', now())
                ->first();

            if ($existing_active) {
                return $this->error('Active Contract Exists.', [], Response::HTTP_CONFLICT);
            }

            $over_lapping = ApplicationAssetVendorContract::where('application_asset_vendor_id', $application_asset_vendor->id)
                ->where(function ($query) use ($request) {
                    $query->whereBetween('start_date', [$request->start_date, $request->end_date])
                        ->orWhereBetween('end_date', [$request->start_date, $request->end_date]);
                })
                ->exists();

            if ($over_lapping) {
                return $this->error('Contract Date Conflict.', [], Response::HTTP_CONFLICT);
            }

            $document_path = null;

            if ($request->hasFile('document_path')) {
                $document_path = $request->file('document_path')->store('contracts', 'public');
            }

            ApplicationAssetVendorContract::create([
                'application_asset_vendor_id' => $request->application_asset_vendor_id,
                'start_date' => $request->start_date,
                'end_date' => $request->end_date,
                'document_path' => $document_path,
                'created_by' => Auth::id(),
                'status' => 1,
            ]);

            return $this->success([], 'Record created.', Response::HTTP_CREATED, true);
        } catch (Exception $e) {
            return $this->error('Error occurred.', $e->getMessage(), Response::HTTP_UNPROCESSABLE_ENTITY);
        }
    }

    public function show(Request $request)
    {
        if (!Gate::allows('View Software Asset') && !Gate::allows('View Service Asset')) {
            return $this->error('Permission Denied', [], 403);
        }

        try {

            $validate = Validator::make($request->all(), [
                'application_asset_vendor_contract_id' => 'required|exists:application_asset_vendor_contracts,id',
            ]);

            if ($validate->fails()) {
                return $this->error('Validation Failed.', $validate->errors(), Response::HTTP_UNPROCESSABLE_ENTITY);
            }

            $contract = ApplicationAssetVendorContract::find($request->application_asset_vendor_contract_id);

            $data = [
                'id' => $contract->id,
                'vendor' => $contract->applicationAssetVendor->vendor->name ?? '',
                'branch' => $contract->applicationAssetVendor->branch->name ?? '',
                'asset' => $contract->applicationAssetVendor->applicationAsset->name ?? '',
                'start_date' => $contract->start_date,
                'end_date' => $contract->end_date,
                'status' => $contract->contract_status,
                'document_path' => $contract->document_path ? asset('storage/' . $contract->document_path) : '',
                'created_by' => $contract->createdBy->name ?? '',
                'Added_date' => $contract->created_at->format('d-m-Y H:i'),
                'renewals' => [],
            ];

            foreach ($contract->applicationAssetVendorContractRenewals as $renewal) {
                $data['renewals'][] = [
                    'id' => $renewal->id,
                    'start_date' => $renewal->start_date,
                    'end_date' => $renewal->end_date,
                    'cost' => $renewal->cost,
                    'descriptions' => $renewal->descriptions,
                    'created_by' => $renewal->createdBy->name ?? '',
                    'added_date' => $renewal->created_at->format('d-m-Y H:i'),
                ];
            }

            return $this->success(['contract_details' => $data], 'Data Retrieved.', Response::HTTP_CREATED, true);
        } catch (Exception $e) {
            return $this->error('Error occurred.', $e->getMessage(), Response::HTTP_UNPROCESSABLE_ENTITY);
        }
    }
}
