<?php

namespace App\Http\Controllers\API\Assets;

use App\Http\Controllers\Controller;
use App\Models\Asset;
use App\Models\AssetAllocation;
use App\Models\AssetBranchPoint;
use App\Models\Branch;
use App\Models\Employee;
use App\Models\Traits\HttpResponses;
use Carbon\Carbon;
use Exception;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Gate;
use Illuminate\Support\Facades\Validator;
use Symfony\Component\HttpFoundation\Response;
use Illuminate\Support\Facades\Http;

class AssetAllocationApiController extends Controller
{
    use HttpResponses;

    public function index(Request $request)
    {
        if (!Gate::allows('View Asset Allocation')) {
            return $this->error('Permission Denied.', [], Response::HTTP_FORBIDDEN);
        }

        $limit = $request->query('limit', 15);
        $search = $request->query('search');

        $query = AssetAllocation::with([
            'asset:id,name,serial_number,model_number,category_id',
            'employee:id,name',
            'branch:id,name'
        ])->latest();

        if (!empty($request->status)) {
            $query->where('status', $request->status);
        }

        if (!empty($search)) {
            $query->whereHas('asset', function ($sub_query1) use ($search) {
                $sub_query1->where('serial_number', 'LIKE', "%{$search}%")
                    ->orWhere('model_number', 'LIKE', "%{$search}%")
                    ->orWhere('name', 'LIKE', "%{$search}%");
            })
                ->orWhereHas('assetStatus', function ($sub_query2) use ($search) {
                    $sub_query2->where('name', 'LIKE', "%{$search}%");
                })
                ->orWhereHas('assetCondition', function ($sub_query2) use ($search) {
                    $sub_query2->where('name', 'LIKE', "%{$search}%");
                });
        }

        if (!empty($request->asset_id)) {
            $query->where('asset_id', $request->asset_id);
        }

        $asset_allocations = $query->paginate($limit);

        $data = [];
        foreach ($asset_allocations as $asset_allocation) {
            $data[] = [
                'id' => $asset_allocation->id,
                'asset' => $asset_allocation->asset->asset_name,
                'serial_number' => $asset_allocation->asset->serial_number,
                'assigned_to' => $asset_allocation->asset->allocated_entity,
                'branch' => $asset_allocation->branch->name,
                'assigned_date' => $asset_allocation->date_assigned,
                'date_unassigned' => $asset_allocation->date_unassigned ?? '',
                'assigned_by' => $asset_allocation->allocatedBy->name,
                'unallocated_by' => $asset_allocation->unallocatedBy->name ?? '',
                'descriptions' => $asset_allocation->descriptions ?? '',
                'reasons' => $asset_allocation->reasons ?? '',
                'status' => $asset_allocation->allocation_status,
            ];
        }

        return $this->success([
            'asset_allocations' => $data,
            'pagination' => [
                'total' => $asset_allocations->total(),
                'current_page' => $asset_allocations->currentPage(),
                'last_page' => $asset_allocations->lastPage(),
                'per_page' => $asset_allocations->perPage()
            ]
        ], 'Data Retrieved.', true);
    }

    public function store(Request $request)
    {
        if (!Gate::allows('Create Asset Allocation')) {
            return $this->error('Permission Denied.', [], Response::HTTP_FORBIDDEN);
        }

        $validate = Validator::make($request->all(), [
            'asset_id' => 'required|exists:assets,id',
            'employee_id' => 'nullable|exists:employees,id',
            'branch_id' => 'nullable|exists:branches,id',
            'assigned_date' => 'nullable|date',
            'reasons' => 'nullable|string',
            'descriptions' => 'nullable|string',
            'action_type' => 'required|in:assign,unassign'
        ]);

        if ($validate->fails()) {
            return $this->error('Validation Failed.', $validate->errors(), Response::HTTP_UNPROCESSABLE_ENTITY);
        }

        try {

            $asset = Asset::find($request->asset_id);
            $current_allocation = $asset->current_allocation;

            if ($request->action_type === 'unassign') {
                if (!$current_allocation) {
                    return $this->error('This asset is not currently assigned.', [], Response::HTTP_UNPROCESSABLE_ENTITY);
                }

                $validate = Validator::make($request->all(), [
                    'reasons' => 'required|string',
                ]);

                if ($validate->fails()) {
                    return $this->error('Validation Failed.', $validate->errors(), Response::HTTP_UNPROCESSABLE_ENTITY);
                }

                $current_allocation->update([
                    'status' => 0,
                    'unassigned_date' => now(),
                    'reasons' => $request->reasons ?? 'Manually unassigned',
                    'unallocated_by' => Auth::id()
                ]);
                return $this->success([], 'Record Updated. ', Response::HTTP_OK, true);
            }

            $validate = Validator::make($request->all(), [
                'employee_id' => 'nullable|exists:employees,id',
                'branch_id' => 'required|exists:branches,id',
                'assigned_date' => 'required|date',
                'reasons' => 'nullable|string',
            ]);

            if ($validate->fails()) {
                return $this->error('Validation Failed.', $validate->errors(), Response::HTTP_UNPROCESSABLE_ENTITY);
            }

            $branch = Branch::find($request->branch_id);

            if ($branch->has_branch_point) {
                $validate = Validator::make($request->all(), [
                    'branch_point_id' => 'required|exists:branch_points,id',
                ]);

                if ($validate->fails()) {
                    return $this->error('Validation Failed.', $validate->errors(), Response::HTTP_UNPROCESSABLE_ENTITY);
                }
            }

            if ($current_allocation) {
                if ($asset->is_employee_assigned && $current_allocation->employee_id == $request->employee_id) {
                    return $this->error('The asset is already assigned to the same employee.', [], Response::HTTP_UNPROCESSABLE_ENTITY);
                }

                if ($asset->is_branch_assigned && $current_allocation->branch_id == $request->branch_id) {
                    return $this->error('The asset is already assigned to the same branch.', [], Response::HTTP_UNPROCESSABLE_ENTITY);
                }

                $current_allocation->update([
                    'status' => 0,
                    'unassigned_date' => now(),
                    'reasons' => $request->reasons ?? 'Reassigned to another entity',
                    'unallocated_by' => Auth::id()
                ]);
            }

            AssetAllocation::create([
                'asset_id' => $request->asset_id,
                'employee_id' => $request->employee_id,
                'branch_id' => $request->branch_id,
                'assigned_date' => Carbon::parse($request->assigned_date)->format('Y-m-d'),
                'unassigned_date' => null,
                'reasons' => null,
                'descriptions' => $request->descriptions,
                'allocated_by' => Auth::id(),
                'status' => 1
            ]);

            if ($asset->branch_id != $request->branch_id) {
                $asset->update(['branch_id' => $request->branch_id]);
            }

            if ($branch->has_branch_point) {
                AssetBranchPoint::create([
                    'asset_id' => $asset->id,
                    'branch_point_id' => $request->branch_point_id,
                    'created_by' => Auth::id(),
                    'status' => 1
                ]);
            }

            return $this->success([], 'Record Created.', Response::HTTP_CREATED, true);
        } catch (Exception $e) {
            return $this->error('Error occurred.', $e->getMessage(), Response::HTTP_UNPROCESSABLE_ENTITY);
        }
    }

    public function selectOption(Request $request)
    {
        $limit = $request->query('limit', 300);
        $search = trim($request->query('search'));

        $employee_query = Employee::select('id', 'name')->orderBy('name')->where('status', 1);
        if (!empty($search)) {
            $employee_query->where('name', 'LIKE', "%{$search}%");
        }

        $employees = $employee_query->limit($limit)->get()->map(function ($employee) {
            return [
                'id' => $employee->id,
                'text' => ucfirst($employee->name),
            ];
        });

        $branch_query = Branch::select('id', 'name')->orderBy('name');
        if (!empty($search)) {
            $branch_query->where('name', 'LIKE', "%{$search}%");
        }
        $branches = $branch_query->limit($limit)->get()->map(function ($branch) {
            return [
                'id' => $branch->id,
                'text' => $branch->name,
            ];
        });

        return $this->success([
            'employees' => $employees,
            'branches' => $branches,
        ], 'Data Retrieved.');
    }

    public function hr()
    {
        Employee::find(35)->update(['email' => 'a.omar@zic.co.tz']);
        Employee::find(38)->update(['email' => 'a.daudi@zic.co.tz']);
        Employee::find(40)->update(['email' => 'e.hangwa@zic.co.tz']);
        Employee::find(42)->update(['email' => 'f.suleiman@zic.co.tz']);
        Employee::find(43)->update(['email' => 'i.habibu@zic.co.tz']);
        Employee::find(44)->update(['email' => 'j.chediel@zic.co.tz']);
        Employee::find(41)->update(['email' => 'f.govin@zic.co.tz']);

        DB::table('employees')
            ->leftJoin('asset_allocations', 'employees.id', '=', 'asset_allocations.employee_id')
            ->whereNull('asset_allocations.employee_id')->where('email', '')
            ->select('employees.*')->delete();

        $url = config('app.hr_url');
        $response = Http::get($url . 'api/v1/auth/employees-option');
        if ($response->successful()) {
            $employees = $response->json();
            foreach ($employees as $employee) {
                $checkIfExist = Employee::where('email', $employee['email'])->first();
                if ($checkIfExist != null) {
                    $checkIfExist->update([
                        'name' => $employee['name'],
                        'email' => $employee['email'],
                        'phone' => $employee['phone'],
                        'designation' => $employee['designation'],
                        'department' => $employee['department'],
                        'branch' => $employee['branch'],
                        'status' => 1,
                        'profile_picture' => $employee['profile_picture']
                    ]);
                } else {
                    Employee::create([
                        'name' => $employee['name'],
                        'email' => $employee['email'],
                        'phone' => $employee['phone'],
                        'designation' => $employee['designation'],
                        'status' => 1,
                        'department' => $employee['department'],
                        'branch' => $employee['branch'],
                        'profile_picture' => $employee['profile_picture']
                    ]);
                }
            }

            $curentEmployees = Employee::get();

            $commonEmployees = collect($employees)->pluck('email')
                ->intersect(collect($curentEmployees)->pluck('email'))
                ->values();

            Employee::whereNotIn('email', $commonEmployees)->update(['status' => 0]);
        }

        return $this->success([], 'Record Synchronized.', Response::HTTP_CREATED, true);
    }
}
