<?php

namespace App\Http\Controllers\API\Assets;

use App\Http\Controllers\Controller;
use App\Models\Asset;
use App\Models\AssetAllocation;
use App\Models\AssetBranchPoint;
use App\Models\AssetCondition;
use App\Models\AssetStatus;
use App\Models\Branch;
use App\Models\Brand;
use App\Models\AssetStateHistory;
use App\Models\BranchPoint;
use App\Models\Category;
use App\Models\Department;
use App\Models\Traits\HttpResponses;
use App\Models\Vendor;
use Carbon\Carbon;
use Exception;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Gate;
use Illuminate\Support\Facades\Validator;
use Illuminate\Validation\Rule;
use Symfony\Component\HttpFoundation\Response;

class AssetApiController extends Controller
{
    use HttpResponses;

    public function index(Request $request)
    {
        if (!Gate::allows('View Asset')) {
            return $this->error('Permission Denied', [], Response::HTTP_FORBIDDEN);
        }

        $limit = $request->query('limit', 15);
        $search = $request->query('search');
        $category_group_ids = $request->category_group_ids;

        $query = Asset::latest();

        if (!empty($category_group_ids)) {
            $validate = Validator::make($request->all(), [
                'category_group_ids' => 'array',
            ]);

            if ($validate->fails()) {
                return $this->error('Validation Failed.', $validate->errors(), Response::HTTP_UNPROCESSABLE_ENTITY);
            }

            $query->whereHas('category.categoryGroup', function ($sub_query) use ($category_group_ids) {
                $sub_query->whereIn('id', $category_group_ids);
            });
        }

        if (!empty($search)) {
            $query->where(function ($sub_query1) use ($search) {
                $sub_query1->where('serial_number', 'LIKE', "%{$search}%")
                    ->orWhere('model_number', 'LIKE', "%{$search}%")
                    ->orWhere('asset_tag', 'LIKE', "%{$search}%")
                    ->orWhereHas('category', function ($sub_query2) use ($search) {
                        $sub_query2->where('name', 'LIKE', "%{$search}%");
                    })
                    ->orWhereHas('brandModel', function ($sub_query2) use ($search) {
                        $sub_query2->where('name', 'LIKE', "%{$search}%");
                    })
                    ->orWhereHas('vendor', function ($sub_query2) use ($search) {
                        $sub_query2->where('name', 'LIKE', "%{$search}%");
                    })
                    ->orWhereHas('assetStatus', function ($sub_query2) use ($search) {
                        $sub_query2->where('name', 'LIKE', "%{$search}%");
                    })
                    ->orWhereHas('assetCondition', function ($sub_query2) use ($search) {
                        $sub_query2->where('name', 'LIKE', "%{$search}%");
                    })
                    ->orWhereHas('assetAllocations.employee',  function ($sub_query2) use ($search) {
                        $sub_query2->where('name', 'LIKE', "%{$search}%");
                    })
                    ->orWhereHas('assetAllocations.branch',  function ($sub_query2) use ($search) {
                        $sub_query2->where('name', 'LIKE', "%{$search}%");
                    });
            });
        }

        $assets = $query->paginate($limit);

        $data = [];
        foreach ($assets as $asset) {
            $data[] = [
                'id' => $asset->id,
                'name' => $asset->asset_name ?? '',
                'serial_number' => $asset->serial_number ?? '',
                'asset_tag' => $asset->asset_tag ?? '',
                'model_number' => $asset->model_number ?? '',
                'vendor' => $asset->vendor->name ?? '',
                'branch' => $asset->branch->name ?? '',
                'brand_model' => $asset->brandModel->brand_model_name ?? '',
                'status' => $asset->assetStatus->name ?? '',
                'capitalization_date' => Carbon::parse($asset->capitalization_date)->format('d-m-Y') ?? '',
                'acquisition_date' => Carbon::parse($asset->acquisition_date)->format('d-m-Y') ?? '',
                'condition' => $asset->assetCondition->name ?? '',
                'is_allocated' => $asset->is_allocated,
                'added_date' => Carbon::parse($asset->created_at)->format('d-m-Y H:i'),
                'assigned_to' => $asset->allocated_entity,
                'condition_id' => $asset->assetCondition->id ?? '',
                'branch_id' => $asset->branch->id ?? '',
                'vendor_id' => $asset->vendor->id ?? '',
                'brand_model_id' => $asset->brandModel->id ?? '',
                'status_id' => $asset->assetStatus->id ?? '',
                'category_id' => $asset->category_id ?? '',
                'cdate' => $asset->capitalization_date ?? '',
                'adate' => $asset->acquisition_date ?? '',
                'condition_id' => $asset->assetCondition->id ?? '',
                'brand' => $asset->brandModel->brand_model_brand_id ?? '',
                'descriptions' => $asset->descriptions
            ];
        }

        return $this->success([
            'assets' => $data,
            'pagination' => [
                'total' => $assets->total(),
                'current_page' => $assets->currentPage(),
                'last_page' => $assets->lastPage(),
                'per_page' => $assets->perPage()
            ]
        ], 'Data Retrieved.', true);
    }

    public function show(Request $request)
    {
        if (!Gate::allows('View Asset')) {
            return $this->error('Permission Denied', [], Response::HTTP_FORBIDDEN);
        }

        $validate = Validator::make($request->all(), [
            'asset_id' => 'required|exists:assets,id'
        ]);

        if ($validate->fails()) {
            return $this->error('Validation Failed.', $validate->errors(), Response::HTTP_UNPROCESSABLE_ENTITY);
        }

        try {
            $asset = Asset::find($request->asset_id);

            if (!$asset) {
                return $this->error('Data not found.', [], Response::HTTP_NOT_FOUND);
            }

            $data = [
                'id' => $asset->id,
                'name' => $asset->asset_name,
                'category_id' => $asset->category_id,
                'serial_number' => $asset->serial_number ?? '',
                'asset_tag' => $asset->asset_tag ?? '',
                'model_number' => $asset->model_number ?? '',
                'vendor_id' => $asset->vendor_id ?? '',
                'vendor_name' => $asset->vendor->name ?? '',
                'branch_id' => $asset->branch_id ?? '',
                'branch_name' => $asset->branch->name ?? '',
                'brand_model_id' => $asset->brand_model_id ?? '',
                'brand_model_name' => $asset->brandModel->brand_model_name ?? '',
                'status_id' => $asset->status_id ?? '',
                'status_name' => $asset->assetStatus->name ?? '',
                'condition_id' => $asset->condition_id ?? '',
                'condition_name' => $asset->assetCondition->name ?? '',
                'is_allocated' => $asset->is_allocated,
                'descriptions' => $asset->descriptions ?? '',
                'assigned_to' => $asset->allocated_entity
            ];

            return $this->success(['asset' => $data], 'Data Retrieved.');
        } catch (Exception $e) {
            return $this->error('Error occurred.', $e->getMessage(), Response::HTTP_UNPROCESSABLE_ENTITY);
        }
    }

    public function store(Request $request)
    {
        if (!Gate::allows('Create Asset')) {
            return $this->error('Permission Denied', [], Response::HTTP_FORBIDDEN);
        }

        $validate = Validator::make($request->all(), [
            'category_id' => 'required|exists:categories,id',
            'branch_id' => 'required|exists:branches,id',
            'condition_id' => 'required|exists:asset_conditions,id',
            'vendor_id' => 'required|exists:vendors,id',
            'status_id' => 'required|exists:asset_statuses,id',
            'brand_model_id' => 'required|exists:brand_models,id',
            'department_id' => 'nullable|exists:departments,id',
            'serial_number' => 'required|string|unique:assets,serial_number',
            'model_number' => 'required|string',
            'asset_tag' => 'nullable|string|unique:assets,asset_tag',
            'capitalization_date' => 'required|date',
            'acquisition_date' => 'nullable|date',
            'descriptions' => 'nullable|string',
        ]);

        if ($validate->fails()) {
            return $this->error('Validation Failed.', $validate->errors(), Response::HTTP_UNPROCESSABLE_ENTITY);
        }

        try {
            DB::beginTransaction();

            $category = Category::find($request->category_id);

            $asset = Asset::create([
                'name' => $category->group_name,
                'category_id' => $request->category_id,
                'branch_id' => $request->branch_id,
                'condition_id' => $request->condition_id,
                'vendor_id' => $request->vendor_id,
                'status_id' => $request->status_id,
                'brand_model_id' => $request->brand_model_id,
                'department_id' => $request->department_id,
                'serial_number' => $request->serial_number,
                'model_number' => $request->model_number,
                'asset_tag' => $request->asset_tag,
                'capitalization_date' => Carbon::parse($request->capitalization_date)->format('Y-m-d'),
                'acquisition_date' => Carbon::parse($request->acquisition_date)->format('Y-m-d'),
                'descriptions' => $request->descriptions,
                'created_by' => Auth::id()
            ]);

            AssetStateHistory::create([
                'asset_id' => $asset->id,
                'status_id' => $request->status_id,
                'condition_id' => $request->condition_id,
                'descriptions' => 'Status is : ' . $asset->assetStatus->name . ' and condition is : ' . $asset->assetCondition->name,
                'created_by' => Auth::id()
            ]);

            if (!$category->is_assignable) {
                AssetAllocation::create([
                    'asset_id' => $asset->id,
                    'employee_id' => null,
                    'branch_id' => $request->branch_id,
                    'status' => 1,
                    'allocated_by' => Auth::id(),
                    'assigned_date' => Carbon::parse($request->capitalization_date)->format('Y-m-d'),
                    'descriptions' => 'Auto Assigned to a registered branch.'
                ]);
            }

            DB::commit();
            return $this->success([], 'Record created.', Response::HTTP_CREATED, true);
        } catch (Exception $e) {
            DB::rollBack();
            return $this->error('Error Occurred.', $e->getMessage(), Response::HTTP_UNPROCESSABLE_ENTITY);
        }
    }

    public function update(Request $request)
    {
        if (!Gate::allows('Update Asset')) {
            return $this->error('Permission Denied', [], Response::HTTP_FORBIDDEN);
        }

        $validate = Validator::make($request->all(), [
            'asset_id' => 'required|exists:assets,id',
            'category_id' => 'required|exists:categories,id',
            'branch_id' => 'nullable|exists:branches,id',
            'condition_id' => 'required|exists:asset_conditions,id',
            'vendor_id' => 'required|exists:vendors,id',
            'status_id' => 'required|exists:asset_statuses,id',
            'brand_model_id' => 'required|exists:brand_models,id',
            'department_id' => 'nullable|exists:departments,id',
            'serial_number' => [
                'required',
                Rule::unique('assets', 'serial_number')->ignore($request->asset_id)
            ],
            'asset_tag' => [
                'nullable',
                Rule::unique('assets', 'asset_tag')->ignore($request->asset_id)
            ],
            'model_number' => 'required|string',
            'capitalization_date' => 'required|date',
            'acquisition_date' => 'nullable|date',
            'descriptions' => 'nullable|string',
        ]);

        if ($validate->fails()) {
            return $this->error('Validation Failed.', $validate->errors(), Response::HTTP_UNPROCESSABLE_ENTITY);
        }

        try {
            $asset = Asset::findOrFail($request->asset_id);

            if ($asset->status_id != $request->status_id && $asset->condition_id != $request->condition_id) {
                AssetStateHistory::create([
                    'asset_id'     => $asset->id,
                    'status_id'    => $request->status_id,
                    'condition_id' => $request->condition_id,
                    'created_by'   => Auth::id(),
                    'descriptions'      => 'Status and condition updated together',
                ]);
            } elseif ($asset->status_id != $request->status_id) {
                $new_status = AssetStatus::find($request->status_id);
                AssetStateHistory::create([
                    'asset_id'     => $asset->id,
                    'status_id'    => $request->status_id,
                    'condition_id' => null,
                    'created_by'   => Auth::id(),
                    'descriptions'      => 'Status updated from ' . $asset->assetStatus->name . ' to ' . $new_status->name,
                ]);
            } elseif ($asset->condition_id != $request->condition_id) {
                $new_condition = AssetCondition::find($request->condition_id);
                AssetStateHistory::create([
                    'asset_id'     => $asset->id,
                    'status_id'    => null,
                    'condition_id' => $request->condition_id,
                    'created_by'   => Auth::id(),
                    'descriptions'      => 'Condition updated from ' . $asset->condition->name . ' to ' . $new_condition->name,
                ]);
            }

            $asset->update([
                'name' => Category::find($request->category_id)->group_name ?? $asset->name,
                'category_id' => $request->category_id,
                'branch_id' => $request->branch_id,
                'condition_id' => $request->condition_id,
                'vendor_id' => $request->vendor_id,
                'status_id' => $request->status_id,
                'brand_model_id' => $request->brand_model_id,
                'department_id' => $request->department_id,
                'serial_number' => $request->serial_number,
                'model_number' => $request->model_number,
                'asset_tag' => $request->asset_tag,
                'capitalization_date' => Carbon::parse($request->capitalization_date)->format('Y-m-d'),
                'acquisition_date' => Carbon::parse($request->acquisition_date)->format('Y-m-d'),
                'descriptions' => $request->descriptions,
            ]);

            return $this->success([], 'Record updated.', Response::HTTP_OK, true);
        } catch (Exception $e) {
            return $this->error('Error Occurred.', $e->getMessage(), Response::HTTP_UNPROCESSABLE_ENTITY);
        }
    }

    public function asset_options()
    {
        $category = Category::orderBy('name')->get(['id', 'name']);
        $condition = AssetCondition::orderBy('name')->get(['id', 'name']);
        $status = AssetStatus::orderBy('name')->get(['id', 'name']);
        $brand = Brand::orderBy('name')->get(['id', 'name']);
        $branch = Branch::orderBy('name')->get(['id', 'name']);
        $department = Department::orderBy('name')->get(['id', 'name']);
        $vendor = Vendor::orderBy('name')->get(['id', 'name']);

        return [
            'categories' => $category,
            'conditions' => $condition,
            'status' => $status,
            'brands' => $brand,
            'branches' => $branch,
            'departments' => $department,
            'vendors' => $vendor,
        ];
    }

    public function assetForSelect(Request $request)
    {
        $limit = $request->query('limit', 100);
        $search = trim($request->query('search'));

        $query = Asset::select('id', 'name', 'serial_number', 'model_number', 'category_id')->orderBy('name');

        if (!empty($search)) {
            $query->where('name', 'LIKE', "%{$search}%")
                ->orWhere('serial_number', 'LIKE', "%{$search}%")
                ->orWhere('model_number', 'LIKE', "%{$search}%");
        }

        $assets = $query->limit($limit)->get();

        $data = [];
        foreach ($assets as $asset) {
            $data[] = [
                'id' => $asset->id,
                'text' => ucfirst($asset->asset_with_number),
            ];
        }

        return $this->success(['assets' => $data], 'Data Retrieved.');
    }

    public function assetDetails(Request $request)
    {
        if (!Gate::allows('View Asset')) {
            return $this->error('Permission Denied', [], Response::HTTP_FORBIDDEN);
        }

        $validate = Validator::make($request->all(), [
            'asset_id' => 'required|exists:assets,id'
        ]);

        if ($validate->fails()) {
            return $this->error('Validation Failed.', $validate->errors(), Response::HTTP_UNPROCESSABLE_ENTITY);
        }

        try {
            $asset = Asset::find($request->asset_id);

            if (!$asset) {
                return $this->error('Data not found.', [], Response::HTTP_NOT_FOUND);
            }

            $asset_detail = [
                'id' => $asset->id,
                'name' => $asset->asset_name,
                'serial_number' => $asset->serial_number ?? '',
                'asset_tag' => $asset->asset_tag ?? '',
                'model_number' => $asset->model_number ?? '',
                'vendor_name' => $asset->vendor->name ?? '',
                'branch_name' => $asset->branch->name ?? '',
                'brand_model_name' => $asset->brandModel->name ?? '',
                'status_name' => $asset->assetStatus->name ?? '',
                'condition_name' => $asset->assetCondition->name ?? '',
                'is_allocated' => $asset->is_allocated,
                'descriptions' => $asset->descriptions ?? '',
                'assigned_to' => $asset->allocated_entity
            ];

            $current_allocation = $asset->assetAllocations
                ->where('status', 1)
                ->map(function ($allocation) {
                    return [
                        'employee' => $allocation->employee->name ?? '',
                        'branch' => $allocation->branch->name ?? '',
                        'assigned_date' => $allocation->date_assigned,
                        'assigned_by' => $allocation->allocatedBy->name,
                    ];
                })
                ->first();

            $allocation_history = $asset->assetAllocations
                ->where('status', 0)
                ->map(function ($allocation) {
                    return [
                        'employee' => $allocation->employee->name ?? '',
                        'branch' => $allocation->branch->name ?? '',
                        'assigned_date' => $allocation->date_assigned,
                        'date_unassigned' => $allocation->date_unassigned,
                        'assigned_by' => $allocation->allocatedBy->name ?? '',
                        'unallocated_by' => $allocation->unallocatedBy->name ?? '',
                        'reasons' => $allocation->reasons ?? '',
                        'status' => $allocation->allocation_status,
                    ];
                });

            $status_history = $asset->assetStateHistories
                ->whereNotNull('status_id')
                ->map(function ($status) {
                    return [
                        'status' => $status->assetStatus->name,
                        'changed_by' => $status->createdBy->name,
                        'changed_at' => Carbon::parse($status->created_at)->format('d-m-Y H:i'),
                    ];
                });

            $condition_history = $asset->assetStateHistories
                ->whereNotNull('condition_id')
                ->map(function ($condition) {
                    return [
                        'condition' => $condition->assetCondition->name ?? '',
                        'changed_by' => $condition->createdBy->name ?? '',
                        'changed_at' => Carbon::parse($condition->created_at)->format('d-m-Y H:i'),
                    ];
                });

            $condition_maintenance = $asset->assetMaintenances
                ->map(function ($maintenance) {
                    return [
                        'maintenance_date' => Carbon::parse($maintenance->date)->format('M d, Y'),
                        'next_date' => $maintenance->plan_date,
                        'remarks' => $maintenance->remarks,
                        'descriptions' => $maintenance->descriptions ?? '', //action taken
                        'fault_description' => $maintenance->fault_description ?? '',
                        'created_by' => $maintenance->createdBy->name,
                        'status' => $maintenance->maintenance_status
                    ];
                });

            return $this->success([
                'asset' => $asset_detail,
                'current_allocation' => $current_allocation ? [$current_allocation] : [],
                'allocation_history' => $allocation_history->values(),
                'status_history' => $status_history->values(),
                'condition_history' => $condition_history->values(),
                'maintenance_history' => $condition_maintenance->values()
            ], 'Data Retrieved.');
        } catch (Exception $e) {
            return $this->error('Error occurred.', $e->getMessage(), Response::HTTP_UNPROCESSABLE_ENTITY);
        }
    }
}
