<?php

namespace App\Http\Controllers\API\Assets;

use App\Http\Controllers\Controller;
use App\Models\AssetMaintenance;
use App\Models\Traits\HttpResponses;
use Carbon\Carbon;
use Exception;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Gate;
use Illuminate\Support\Facades\Validator;
use Symfony\Component\HttpFoundation\Response;

class AssetMaintenanceApiController extends Controller
{
    use HttpResponses;

    public function index(Request $request)
    {
        if (!Gate::allows('View Asset Maintenance')) {
            return $this->error('Permission Denied', [], Response::HTTP_FORBIDDEN);
        }

        $limit = $request->query('limit', 15);
        $search = $request->query('search');

        $query = AssetMaintenance::with([
            'asset:id,name,serial_number,model_number,category_id',
        ])
            ->latest();

        if (!empty($request->status)) {
            $query->where('status', $request->status);
        }

        if (!empty($search)) {
            $query->whereHas('asset', function ($sub_query1) use ($search) {
                $sub_query1->where('serial_number', 'LIKE', "%{$search}%")
                    ->orWhere('model_number', 'LIKE', "%{$search}%")
                    ->orWhere('name', 'LIKE', "%{$search}%");
            })
                ->orWhere('status', 'LIKE', "%{$search}%");
        }

        if (!empty($request->asset_id)) {
            $query->where('asset_id', $request->asset_id);
        }

        $asset_maintenances = $query->paginate($limit);

        $data = [];
        foreach ($asset_maintenances as $asset_maintenance) {
            $data[] = [
                'id' => $asset_maintenance->id,
                'asset_name' => $asset_maintenance->asset->asset_name,
                'model_number' => $asset_maintenance->asset->model_number ?? '',
                'date' => Carbon::parse($asset_maintenance->date)->format('M d, Y'),
                'next_date' => $asset_maintenance->plan_date,
                'added_date' => Carbon::parse($asset_maintenance->created_at)->format('d-m-Y H:i'),
                'added_by' => $asset_maintenance->createdBy->name,
                'status' => $asset_maintenance->maintenance_status,
                'descriptions' => $asset_maintenance->descriptions,
                'fault_description' => $asset_maintenance->fault_description,
                'remarks' => $asset_maintenance->remarks,
            ];
        }

        return $this->success([
            'asset_maintenances' => $data,
            'pagination' => [
                'total' => $asset_maintenances->total(),
                'current_page' => $asset_maintenances->currentPage(),
                'last_page' => $asset_maintenances->lastPage(),
                'per_page' => $asset_maintenances->perPage()
            ]
        ], 'Data Retrieved.', true);
    }

    public function store(Request $request)
    {
        if (!Gate::allows('Create Asset Maintenance')) {
            return $this->error('Permission Denied.', [], Response::HTTP_FORBIDDEN);
        }

        $validate = Validator::make($request->all(), [
            'asset_id' => 'required|exists:assets,id',
            'maintenance_date' => 'required|date',
            'next_date' => 'nullable|date',
            'remarks' => 'nullable|string',
            'descriptions' => 'required|string',
            'status' => 'required|in:COMPLETED,FAILED',
            'fault_description' => 'required|string',
        ]);

        if ($validate->fails()) {
            return $this->error('Validation Failed.', $validate->errors(), Response::HTTP_UNPROCESSABLE_ENTITY);
        }

        try {
            AssetMaintenance::create([
                'asset_id' => $request->asset_id,
                'date' => Carbon::parse($request->maintenance_date)->format('Y-m-d'),
                'next_date' => Carbon::parse($request->next_date)->format('Y-m-d'),
                'remarks' => $request->remarks,
                'descriptions' => $request->descriptions,
                'fault_description' => $request->fault_description,
                'created_by' => Auth::id(),
                'status' => $request->status,
            ]);

            return $this->success([], 'Record created.', Response::HTTP_CREATED, true);
        } catch (Exception $e) {
            return $this->error('Error occurred.', $e->getMessage(), Response::HTTP_UNPROCESSABLE_ENTITY);
        }
    }

    public function show(Request $request)
    {
        if (!Gate::allows('View Asset Maintenance')) {
            return $this->error('Permission Denied', [], Response::HTTP_FORBIDDEN);
        }

        $validate = Validator::make($request->all(), [
            'asset_maintenance_id' => 'required|exists:asset_maintenances,id'
        ]);

        if ($validate->fails()) {
            return $this->error('Validation Failed.', $validate->errors(), Response::HTTP_UNPROCESSABLE_ENTITY);
        }

        try {
            $asset_maintenance = AssetMaintenance::find($request->asset_maintenance_id);

            if (!$asset_maintenance) {
                return $this->error('Data not found.', [], Response::HTTP_NOT_FOUND);
            }

            $data = [
                'id' => $asset_maintenance->id,
                'asset_name' => $asset_maintenance->asset->asset_name,
                'model_number' => $asset_maintenance->asset->model_number ?? '',
                'asset_id' => $asset_maintenance->asset->id,
                'date' => Carbon::parse($asset_maintenance->date)->format('M d, Y'),
                'next_date' => $asset_maintenance->plan_date,
                'status' => $asset_maintenance->maintenance_status,
                'remarks' => $asset_maintenance->remarks,
                'descriptions' => $asset_maintenance->descriptions ?? '',
                'fault_description' => $asset_maintenance->fault_description ?? '',
                'added_by' => $asset_maintenance->createdBy->name,
            ];

            return $this->success(['asset_maintenance' => $data], 'Data Retrieved.');
        } catch (Exception $e) {
            return $this->error('Error occurred.', $e->getMessage(), Response::HTTP_UNPROCESSABLE_ENTITY);
        }
    }
}
