<?php

namespace App\Http\Controllers\API\Assets;

use App\Http\Controllers\Controller;
use App\Models\Asset;
use App\Models\AssetStateHistory;
use App\Models\Traits\HttpResponses;
use Exception;
use Illuminate\Http\Request;
use Illuminate\Support\Carbon;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Gate;
use Illuminate\Support\Facades\Validator;
use Symfony\Component\HttpFoundation\Response;

class AssetStateHistoryApiController extends Controller
{
    use HttpResponses;

    public function index(Request $request)
    {
        if (!Gate::allows('View Asset State History')) {
            return $this->error('Permission Denied.', [], Response::HTTP_FORBIDDEN);
        }

        $limit = $request->query('limit', 15);
        $search = $request->query('search');

        $query = AssetStateHistory::with([
            'asset:id,name,serial_number,model_number,category_id',
            'assetStatus:id,name',
            'assetCondition:id,name',
        ])->latest();

        if (!empty($request->asset_id)) {
            $query->where('asset_id', $request->asset_id);
        }

        if (!empty($search)) {
            $query->whereHas('asset', function ($sub_query1) use ($search) {
                $sub_query1->where('serial_number', 'LIKE', "%{$search}%")
                    ->orWhere('model_number', 'LIKE', "%{$search}%")
                    ->orWhere('name', 'LIKE', "%{$search}%");
            })
                ->orWhereHas('assetStatus', function ($sub_query2) use ($search) {
                    $sub_query2->where('name', 'LIKE', "%{$search}%");
                })
                ->orWhereHas('assetCondition', function ($sub_query2) use ($search) {
                    $sub_query2->where('name', 'LIKE', "%{$search}%");
                });
        }

        $asset_state_histories = $query->paginate($limit);

        $data = [];
        foreach ($asset_state_histories as $asset_state) {
            $data[] = [
                'id' => $asset_state->id,
                'asset_name' => $asset_state->asset->asset_with_number,
                'status' => $asset_state->assetStatus->name ?? '',
                'condition' => $asset_state->assetCondition->name ?? '',
                'descriptions' => $asset_state->descriptions ?? '',
                'changed_by' => $asset_state->createdBy->name,
                'changed_at' => Carbon::parse($asset_state->changed_at)->format('d-m-Y H:i'),
            ];
        }

        return $this->success([
            'asset_state_histories' => $data,
            'pagination' => [
                'total' => $asset_state_histories->total(),
                'current_page' => $asset_state_histories->currentPage(),
                'last_page' => $asset_state_histories->lastPage(),
                'per_page' => $asset_state_histories->perPage()
            ]
        ], 'Data Retrieved.', true);
    }

    public function update(Request $request)
    {
        if (!Gate::allows('Update Asset State History')) {
            return $this->error('Permission Denied.', [], Response::HTTP_FORBIDDEN);
        }

        $validate = Validator::make($request->all(), [
            'asset_id' => 'required|exists:assets,id',
            'action_type' => 'required|in:status,condition',
            'value_id' => 'required|integer'
        ]);

        if ($validate->fails()) {
            return $this->error('Validation Failed.', $validate->errors(), Response::HTTP_UNPROCESSABLE_ENTITY);
        }

        try {
            $asset = Asset::findOrFail($request->asset_id);

            if ($request->action_type === 'status') {
                if ($asset->status_id == $request->value_id) {
                    return $this->error('Status is already the same.', [], Response::HTTP_UNPROCESSABLE_ENTITY);
                }

                $asset->update(['status_id' => $request->value_id]);

                AssetStateHistory::create([
                    'asset_id'     => $asset->id,
                    'status_id'    => $request->value_id,
                    'condition_id' => null,
                    'created_by'   => Auth::id(),
                    'descriptions'      => 'Status changed.'
                ]);
            }

            if ($request->action_type === 'condition') {
                if ($asset->condition_id == $request->value_id) {
                    return $this->error('Condition is already the same.',  [], Response::HTTP_UNPROCESSABLE_ENTITY);
                }

                $asset->update(['condition_id' => $request->value_id]);

                AssetStateHistory::create([
                    'asset_id'     => $asset->id,
                    'status_id'    => null,
                    'condition_id' => $request->value_id,
                    'created_by'   => Auth::id(),
                    'descriptions'      => 'Condition changed.'
                ]);
            }

            return $this->success([], 'Record Updated. ', Response::HTTP_CREATED, true);
        } catch (Exception $e) {
            return $this->error('Error Occurred. ', $e->getMessage(), Response::HTTP_UNPROCESSABLE_ENTITY);
        }
    }
}
