<?php

namespace App\Http\Controllers\API;

use App\Models\Traits\HttpResponses;
use Exception;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Gate;
use Illuminate\Support\Facades\Validator;
use Symfony\Component\HttpFoundation\Response;
use App\Http\Controllers\API\BaseController as BaseController;
use App\Http\Requests\LoginRequest;
use App\Models\User;
use Illuminate\Support\Facades\Hash;
use Spatie\Permission\Models\Permission;
use Spatie\Permission\Models\Role;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Http;
use PHPOpenSourceSaver\JWTAuth\Facades\JWTAuth;

class AuthController extends BaseController
{

    use HttpResponses;

    /**
     * Register a User.
     *
     * @return \Illuminate\Http\JsonResponse
     */
    public function register(Request $request){
        $validate = Validator::make($request->all(), [
            'name' => 'required',
            'email' => 'required|unique:users,email',
            'roles' => 'required',
        ]);

        if ($validate->fails()) {
            return $this->error('Validation Failed.', $validate->errors(), Response::HTTP_UNPROCESSABLE_ENTITY);
        }

        try {
            $user = User::create([
                'name' => $request->name,
                'email' => $request->email,
                'is_active' => 1,
                'attemps' => 0,
                'password' => Hash::make('User@2025')
            ]);
            $user->assignRole($request->roles);
            return $this->success($user, 'User created.', Response::HTTP_CREATED, true);
        } catch (Exception $e) {
            return $this->error('Error Occurred.', $e->getMessage(), Response::HTTP_UNPROCESSABLE_ENTITY);
        }
    }

    /**
     * Get a JWT via given credentials.
     *
     * @return \Illuminate\Http\JsonResponse
     */
    public function login(LoginRequest $request){
        try {
            $request->validated($request->all());

            $token = JWTAuth::attempt(['email' => $request->email, 'password' => $request->password, 'is_active' => 1]);

            if (!$token) {
                return $this->error('Invalid credentials', [], 401);
            }

            $user = JWTAuth::user();
            $roles = $user->getRoleNames();

            $permissions = [];
            foreach ($user->getAllPermissions() as $permission) {
                $permissions[] = [
                    'name' => $permission->name
                ];
            }

            $url = config('app.hr_url');
            $image = $url.'/public/images/user.png';
            $response = Http::get($url . 'api/v1/auth/employee-profile-photo/'.$user->email);
            if ($response->successful()) {
                $result = $response->json();
                $image = $url.'/public/images/'.$result['profile_picture'];
            }

            $data = $this->respondWithToken($token, array(
                'user' => [
                    'id' => $user->id,
                    'name' => $user->name,
                    'phone' => $user->phone,
                    'email' => $user->email,
                    'status' => $user->is_active,
                    'image' =>  $image
                ],
                'roles' => $roles,
                'permissions' => $permissions,
            ));

            return $this->success($data, 'Login successful.');
        } catch (Exception $e) {
            return $this->error([], $e->getMessage(), Response::HTTP_UNPROCESSABLE_ENTITY);
        }
    }

    /**
     * Log the user out (Invalidate the token).
     *
     * @return \Illuminate\Http\JsonResponse
     */
    public function logout()
    {
        try {
            $token = JWTAuth::getToken();
            JWTAuth::invalidate($token);
            return $this->success([], 'You have successfully logged out.');
        } catch (Exception $e) {
            return $this->error([], $e->getMessage(), Response::HTTP_INTERNAL_SERVER_ERROR);
        }
    }

    /**
     * Get the token array structure.
     *
     * @param  string $token
     *
     * @return \Illuminate\Http\JsonResponse
     */
    protected function respondWithToken($token, $user)
    {
        return [
            'access_token' => $token,
            'token_type' => 'bearer',
            'expires_in' => Auth::factory()->getTTL() * 60,
            'auth' => $user
        ];
    }

    public function index(Request $request){
        if (!Gate::allows('User Management')) {
            return $this->error('Permission Denied', [], Response::HTTP_FORBIDDEN);
        }

       try {
            $limit = $request->query('limit', 15);
            $search = $request->query('search');

            $query = User::with(['roles','permissions'])->latest();

            if (!empty($search)) {
                $query->where(function ($sub_query1) use ($search) {
                    $sub_query1->where('name', 'LIKE', "%{$search}%")->orWhere('email', 'LIKE', "%{$search}%");
                });
            }

            $users = $query->paginate($limit);

            $data = [];
            foreach ($users as $user) {
                $data[] = [
                    'id' => $user->id,
                    'name' => $user->name ?? '',
                    'email' => $user->email ?? '',
                    'roles' => $user->roles->pluck('name'),
                    'status_name' => $user->is_active == 1 ? 'Active' : 'Inactive',
                    'status' => $user->is_active,
                    'permissions' => $user->getAllPermissions()->map(function ($permission) {
                        return [
                            'id' => $permission->id,
                            'name' => $permission->name,
                        ];
                    }),
                ];
            }

            return $this->success([
                'users' => $data,
                'pagination' => [
                    'total' => $users->total(),
                    'current_page' => $users->currentPage(),
                    'last_page' => $users->lastPage(),
                    'per_page' => $users->perPage()
                ]
            ], 'Data Retrieved.', true);
        } catch (\Throwable $th) {
            return $this->error('Error Occurred.', $th->getMessage(), Response::HTTP_UNPROCESSABLE_ENTITY);
       }
    }

    public function update(Request $request){
        $validate = Validator::make($request->all(), [
            'name' => 'required',
            'email' => 'required|email|unique:users,email,' . $request->id,
            'roles' => 'required',
            'id' => 'required'
        ]);

        if ($validate->fails()) {
            return $this->error('Validation Failed.', $validate->errors(), Response::HTTP_UNPROCESSABLE_ENTITY);
        }

        try {
            $user = User::find($request->id);
            $user->update([
                'name' => $request->name,
                'email' => $request->email,
            ]);
            $user->syncRoles($request->roles);
            return $this->success($user, 'User Updated.', Response::HTTP_CREATED, true);
        } catch (Exception $e) {
            return $this->error('Error Occurred.', $e->getMessage(), Response::HTTP_UNPROCESSABLE_ENTITY);
        }
    }

    public function user_status(Request $request){
        $validate = Validator::make($request->all(), [
            'status' => 'required',
            'id' => 'required'
        ]);

        if ($validate->fails()) {
            return $this->error('Validation Failed.', $validate->errors(), Response::HTTP_UNPROCESSABLE_ENTITY);
        }

        try {
            $user = User::find($request->id);
            if($request->status == 1){
                $user->update([
                    'is_active' => 0,
                ]);
                return $this->success($user, 'User Updated.', Response::HTTP_CREATED, true);
            }
            $user->update([
                'is_active' => 1,
            ]);
            return $this->success($user, 'User Updated.', Response::HTTP_CREATED, true);
        } catch (Exception $e) {
            return $this->error('Error Occurred.', $e->getMessage(), Response::HTTP_UNPROCESSABLE_ENTITY);
        }
    }

    public function permissions(){
        $permissions = [];
        $data = Permission::orderBy('name')->get();
        foreach ($data as $permission) {
            $data = array(
                "id" => $permission->id,
                "name" => $permission->name,
                "isSelected" => false
            );
            $permissions[] = $data;
        }
        return response()->json($permissions);
    }

    public function roles(){
        $response = [];
        $roles = Role::orderBy('name')->get();
        foreach ($roles as $role) {
            $rolePermissions = Permission::join('role_has_permissions', 'role_has_permissions.permission_id', 'permissions.id')->where('role_has_permissions.role_id', $role->id)->get();
            $permissions = [];
            foreach ($rolePermissions as $permission) {
                $data = array(
                    "id" => $permission->id,
                    "name" => $permission->name,
                    "isSelected" => true,
                );
                $permissions[] = $data;
            }

            $rolesList = array(
                "id" => $role->id,
                "name" => $role->name,
                "isSelected" => false,
                "permissions" => $permissions
            );
            $response[] = $rolesList;
        }
        return response()->json($response);
    }

    public function updateRole(Request $request){
        $validate = Validator::make($request->all(), [
            'name' => 'required|unique:roles,name,' . $request->id,
            'permissions' => 'required',
            'id' => 'required'
        ]);

        if ($validate->fails()) {
            return $this->error('Validation Failed.', $validate->errors(), Response::HTTP_UNPROCESSABLE_ENTITY);
        }

        try {
            $role = Role::find($request->id);
            $role->update(['name' => $request->name]);
            $role->syncPermissions($request->permissions);

            return $this->success($role, 'Role updated.', Response::HTTP_CREATED, true);
        } catch (Exception $e) {
            return $this->error('Error Occurred.', $e->getMessage(), Response::HTTP_UNPROCESSABLE_ENTITY);
        }
    }

    public function createRole(Request $request){
        $validate = Validator::make($request->all(), [
            'name' => 'required|unique:roles,name',
            'permissions' => 'required',
        ]);

        if ($validate->fails()) {
            return $this->error('Validation Failed.', $validate->errors(), Response::HTTP_UNPROCESSABLE_ENTITY);
        }

        try {
            $role = Role::create(['name' => $request->name]);
            $role->syncPermissions($request->permissions);
            return $this->success($role, 'Role created.', Response::HTTP_CREATED, true);
        } catch (Exception $e) {
            return $this->error('Error Occurred.', $e->getMessage(), Response::HTTP_UNPROCESSABLE_ENTITY);
        }
    }

    public function sync_authorization(){
        try {
            $user = JWTAuth::parseToken()->authenticate();
            $token = JWTAuth::fromUser($user);
            $roles = $user->getRoleNames();

            $permissions = [];
            foreach ($user->getAllPermissions() as $permission) {
                $permissions[] = [
                    'name' => $permission->name
                ];
            }

            $data = $this->respondWithToken($token, array(
                'user' => [
                    'id' => $user->id,
                    'name' => $user->name,
                    'phone' => $user->phone,
                    'email' => $user->email,
                    'status' => $user->is_active,
                ],
                'roles' => $roles,
                'permissions' => $permissions,
            ));
            return $data;
        } catch (Exception $e) {
            return response()->json(['message' => 'Token is invalid or expired'], 401);
        }
    }

    public function syncUsers()
    {
        try {
            if (!$this->checkInternet()) {
                return array(
                    'status' => 405,
                    'message' => 'No Internet Connection, You cant not authenticated.'
                );
            }
            $ch = curl_init();
            curl_setopt($ch, CURLOPT_URL, config('app.hr_url') . "/api/v1/auth/employees-option");
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, 1);
            curl_setopt($ch, CURLOPT_CUSTOMREQUEST, "GET");
            $headers = array();
            $headers[] = "Accept: application/json";
            curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
            $result = curl_exec($ch);
            if (curl_errno($ch)) {
                echo 'Error:' . curl_error($ch);
            }
            curl_close($ch);

            $response = json_decode($result);
            $active_employees = [];
            foreach ($response as $item) {
                array_push($active_employees, $item->email);
            }

            $current_employees = User::get();
            foreach ($current_employees as $item) {
                if (!in_array($item->email, $active_employees)) {
                    User::where('email', $item->email)->update([
                        'isActive' => 0
                    ]);
                } else {
                    User::where('email', $item->email)->update([
                        'isActive' => 1
                    ]);
                }
            }

            foreach ($response as $item) {
                $password = Str::random(12);
                $employee = user::where('email', $item->email)->get()->first();
                if ($employee == null) {
                    $user = User::create([
                        'name' => $item->name,
                        'email' => $item->email,
                        'password' => Hash::make($password)
                    ]);
                    if ($item->email == 'm.mohamed@zic.co.tz') {
                        $user->assignRole('Root');
                    } else if ($item->email == 's.abdulla@zic.co.tz') {
                        $user->assignRole('Admin');
                    } else {
                        $user->assignRole('User');
                    }
                }
            }
            return array(
                'status' => 200,
                'message' => 'Successfully'
            );
        } catch (\Throwable $th) {
            return array(
                'status' => 500,
                'message' => $th->getMessage()
            );
        }
        return json_decode($result);
    }

    public function checkInternet()
    {
        try {
            // Use a HEAD request to avoid downloading the full page
            $response = Http::timeout(5)->head('https://www.google.com');
            if ($response->successful()) {
                return true; // Internet is working
            }
        } catch (\Exception $e) {
            return false; // No internet
        }
        return false;
    }

    public function test_()
    {
        return Hash::make('Zic$2024');
    }
}
