<?php

namespace App\Http\Controllers\API\Operations;

use App\Http\Controllers\Controller;
use App\Models\ApplicationAsset;
use App\Models\Asset;
use App\Models\AssetAllocation;
use App\Models\Traits\HttpResponses;

class DashboardApiController extends Controller
{
    use HttpResponses;

    public function index()
    {
        $greeting = 'Welcome to Zanzibar Insurance Corporation ICT Asset Management Information System.';
        $today = now()->format('l jS \of F Y');

        $total_assets = Asset::count();

        $software_asset = ApplicationAsset::where('status', true)
            ->where('type', 'SOFTWARE')
            ->count();

        $service_asset = ApplicationAsset::where('status', true)
            ->where('type', 'SERVICE')
            ->count();

        $allocated_asset = AssetAllocation::where('status', true)->count();

        $allocated_asset_ids = AssetAllocation::pluck('asset_id')->toArray();
        $unallocated_asset = Asset::whereNotIn('id', $allocated_asset_ids)->count();

        $expired_license = 2;
        $lose_and_damage_asset = 10;

        $employee_has_asset = AssetAllocation::where('status', true)
            ->whereNotNull('employee_id')
            ->count();

        $allowed_application = 8;

        $asset_by_category_chart = Asset::with('category')
            ->get()
            ->groupBy(function ($asset) {
                return optional($asset->category)->name ?? 'Uncategorized';
            })
            ->map->count();

        $asset_by_group = Asset::with('category.categoryGroup')
            ->get()
            ->groupBy(function ($asset) {
                return optional(optional($asset->category)->categoryGroup)->name ?? 'Ungrouped';
            })
            ->map(function ($group) {
                $group_name = optional(optional($group->first()->category)->categoryGroup)->name ?? 'Ungrouped';
                return [
                    'name' => $group_name,
                    'total_asset' => $group->count(),
                ];
            })
            ->values();

        $asset_by_branch = Asset::whereNotNull('branch_id')
            ->with('branch')
            ->get()
            ->groupBy(function ($asset) {
                return optional($asset->branch)->name ?? 'Unbranched';
            })
            ->map(function ($assets, $branch_name) {
                return [
                    'name' => $branch_name,
                    'total_asset' => $assets->count(),
                ];
            })
            ->values();

        $asset_by_status = Asset::whereNotNull('status_id')
            ->with('assetStatus')
            ->get()
            ->groupBy(function ($asset) {
                return optional($asset->assetStatus)->name ?? 'Not Defined';
            })
            ->map(function ($assets, $status_name) {
                return [
                    'name' => $status_name,
                    'total_asset' => $assets->count(),
                ];
            })
            ->values();

        $asset_by_condition = Asset::whereNotNull('status_id')
            ->with('assetCondition')
            ->get()
            ->groupBy(function ($asset) {
                return optional($asset->assetCondition)->name ?? 'Not Defined';
            })
            ->map(function ($assets, $condition_name) {
                return [
                    'name' => $condition_name,
                    'total_asset' => $assets->count(),
                ];
            })
            ->values();

        $data = [
            'greeting' => $greeting,
            'today' => $today,
            'total_assets' => $total_assets,
            'software_assets' => $software_asset,
            'service_assets' => $service_asset,
            'allocated_asset' => $allocated_asset,
            'unallocated_asset' => $unallocated_asset,
            'expired_license' => $expired_license,
            'lose_and_damage_asset' => $lose_and_damage_asset,
            'employee_has_asset' => $employee_has_asset,
            'allowed_application' => $allowed_application,
            'asset_by_category_chart' => $asset_by_category_chart,
            'asset_by_group_table' => $asset_by_group,
            'asset_by_branch' => $asset_by_branch,
            'asset_by_status' => $asset_by_status,
            'asset_by_condition' => $asset_by_condition,
            'footer' => 'Powered by ZIC-ICT-UNITY © ' . date('Y'),
            'backend_app_version' => config('app.version'),
        ];

        return $this->success([
            'dashboard' => $data,
        ], 'Data Retrieved.', true);
    }
}
