<?php

namespace App\Http\Controllers\API\Setups;

use App\Http\Controllers\Controller;
use App\Models\AssetCondition;
use App\Models\Traits\HttpResponses;
use Carbon\Carbon;
use Exception;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Gate;
use Illuminate\Support\Facades\Validator;
use Symfony\Component\HttpFoundation\Response;

class AssetConditionApiController extends Controller
{
    use HttpResponses;

    public function index(Request $request)
    {
        if (!Gate::allows('View Asset Condition')) {
            return $this->error('Permission Denied', [], Response::HTTP_FORBIDDEN);
        }

        $limit = $request->query->get('limit', 15);
        $search = $request->query->get('search');

        $query = AssetCondition::select('id', 'name')->latest();

        if (!empty($search)) {
            $query->where('name', 'LIKE', "%{$search}%");
        }

        $conditions = $query->paginate($limit);


        $data = [];

        foreach ($conditions as $condition) {
            $data[] = [
                'id' => $condition->id,
                'name' => $condition->name,
                'added_date' =>  Carbon::parse($condition->created_at)->format('d:m:Y h:i')
            ];
        }

        return $this->success([
            'asset_conditions' => $data,
            'pagination' => [
                'total' => $conditions->total(),
                'current_page' => $conditions->currentPage(),
                'last_page' => $conditions->lastPage(),
                'per_page' => $conditions->perPage()
            ]
        ], 'Data Retrieved.', true);
    }

    public function show(Request $request)
    {
        if (!Gate::allows('View Asset Condition')) {
            return $this->error('Permission Denied', [], Response::HTTP_FORBIDDEN);
        }

        $validate = Validator::make($request->all(), [
            'asset_condition_id' => 'required|exists:asset_conditions,id',
        ]);

        if ($validate->fails()) {
            return $this->error('Validation Failed.', $validate->errors(), Response::HTTP_UNPROCESSABLE_ENTITY);
        }

        try {
            $condition = AssetCondition::findOrFail($request->asset_condition_id);

            $data = [
                'id' => $condition->id,
                'name' => $condition->name,
            ];

            return $this->success(['asset_condition' => $data], 'Data Retrieved.');
        } catch (Exception $e) {
            return $this->error('Error occurred.', $e->getMessage(), Response::HTTP_UNPROCESSABLE_ENTITY);
        }
    }

    public function store(Request $request)
    {
        if (!Gate::allows('Create Asset Condition')) {
            return $this->error('Permission Denied', [], Response::HTTP_FORBIDDEN);
        }

        $validate = Validator::make($request->all(), ['name' => 'required|string|unique:asset_conditions,name']);
        if ($validate->fails()) {
            return $this->error('Validation Failed.', $validate->errors(), Response::HTTP_UNPROCESSABLE_ENTITY);
        }

        try {
            AssetCondition::create(['name' => $request->name]);
            return $this->success([], 'Data created.', Response::HTTP_CREATED, true);
        } catch (Exception $e) {
            return $this->error('Error Occurred.', $e->getMessage(), Response::HTTP_UNPROCESSABLE_ENTITY);
        }
    }

    public function update(Request $request)
    {
        if (!Gate::allows('Update Asset Condition')) {
            return $this->error('Permission Denied', [], Response::HTTP_FORBIDDEN);
        }

        $validate = Validator::make($request->all(), [
            'asset_condition_id' => 'required|exists:asset_conditions,id',
            'name' => 'required|string|unique:asset_conditions,name,' . $request->asset_condition_id,
        ]);

        if ($validate->fails()) {
            return $this->error('Validation Failed.', $validate->errors(), Response::HTTP_UNPROCESSABLE_ENTITY);
        }

        try {
            $condition = AssetCondition::findOrFail($request->asset_condition_id);
            $condition->update(['name' => $request->name]);

            return $this->success([], 'Data updated.', Response::HTTP_OK, true);
        } catch (Exception $e) {
            return $this->error('Error Occurred.', $e->getMessage(), Response::HTTP_UNPROCESSABLE_ENTITY);
        }
    }

    public function conditionForSelect(Request $request)
    {
        $limit = $request->query('limit', 50);
        $search = trim($request->query('search'));

        $query = AssetCondition::select('id', 'name')->orderBy('name');

        if (!empty($search)) {
            $query->where('name', 'LIKE', "%{$search}%");
        }

        $conditions = $query->limit($limit)->get();

        $data = [];
        foreach ($conditions as $condition) {
            $data[] = [
                'id' => $condition->id,
                'text' => ucfirst($condition->name),
            ];
        }

        return $this->success(['conditions' => $data], 'Data Retrieved.');
    }
}
