<?php

namespace App\Http\Controllers\API\Setups;

use App\Http\Controllers\Controller;
use App\Models\AssetStatus;
use App\Models\Traits\HttpResponses;
use Carbon\Carbon;
use Exception;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Gate;
use Illuminate\Support\Facades\Validator;
use Symfony\Component\HttpFoundation\Response;

class AssetStatusApiController extends Controller
{
    use HttpResponses;

    public function index(Request $request)
    {
        if (!Gate::allows('View Asset Status')) {
            return $this->error('Permission Denied', [], Response::HTTP_FORBIDDEN);
        }

        $limit = $request->query->get('limit', 15);
        $search = $request->query->get('search');

        $query = AssetStatus::select('id', 'name')->latest();

        if (!empty($search)) {
            $query->where('name', 'LIKE', "%{$search}%");
        }

        $statuses = $query->paginate($limit);


        $data = [];
        foreach ($statuses as $status) {
            $data[] = [
                'id' => $status->id,
                'name' => $status->name,
                'added_date' => Carbon::parse($status->created_at)->format('d:m:Y h:i')
            ];
        }

        return $this->success([
            'asset_statuses' => $data,
            'pagination' => [
                'total' => $statuses->total(),
                'current_page' => $statuses->currentPage(),
                'last_page' => $statuses->lastPage(),
                'per_page' => $statuses->perPage()
            ]
        ], 'Data Retrieved.', true);
    }

    public function show(Request $request)
    {
        if (!Gate::allows('View Asset Status')) {
            return $this->error('Permission Denied.', [], Response::HTTP_FORBIDDEN);
        }

        $validate = Validator::make($request->all(), [
            'asset_status_id' => 'required|exists:asset_statuses,id',
        ]);

        if ($validate->fails()) {
            return $this->error('Validation Failed.', $validate->errors(), Response::HTTP_UNPROCESSABLE_ENTITY);
        }

        try {
            $status = AssetStatus::findOrFail($request->asset_status_id);

            $data = [
                'id' => $status->id,
                'name' => $status->name
            ];

            return $this->success(['asset_status' => $data], 'Data Retrieved.');
        } catch (Exception $e) {
            return $this->error('Error occurred.', $e->getMessage(), Response::HTTP_UNPROCESSABLE_ENTITY);
        }
    }

    public function store(Request $request)
    {
        if (!Gate::allows('Create Asset Status')) {
            return $this->error('Permission Denied.', [], Response::HTTP_FORBIDDEN);
        }

        $validate = Validator::make($request->all(), [
            'name' => 'required|string|unique:asset_statuses,name',
        ]);

        if ($validate->fails()) {
            return $this->error('Validation Failed.', $validate->errors(), Response::HTTP_UNPROCESSABLE_ENTITY);
        }

        try {
            AssetStatus::create(['name' => $request->name]);
            return $this->success([], 'Data created.', Response::HTTP_CREATED, true);
        } catch (Exception $e) {
            return $this->error('Error Occurred.', $e->getMessage(), Response::HTTP_UNPROCESSABLE_ENTITY);
        }
    }

    public function update(Request $request)
    {
        if (!Gate::allows('Update Asset Status')) {
            return $this->error('Permission Denied.', [], Response::HTTP_FORBIDDEN);
        }

        $validate = Validator::make($request->all(), [
            'asset_status_id' => 'required|exists:asset_statuses,id',
            'name' => 'required|string|unique:asset_statuses,name,' . $request->asset_status_id,
        ]);

        if ($validate->fails()) {
            return $this->error('Validation Failed.', $validate->errors(), Response::HTTP_UNPROCESSABLE_ENTITY);
        }

        try {
            $status = AssetStatus::findOrFail($request->asset_status_id);
            $status->update(['name' => $request->name]);

            return $this->success([], 'Data updated.', Response::HTTP_OK, true);
        } catch (Exception $e) {
            return $this->error('Error Occurred.', $e->getMessage(), Response::HTTP_UNPROCESSABLE_ENTITY);
        }
    }

    public function statusForSelect(Request $request)
    {
        $limit = $request->query('limit', 50);
        $search = trim($request->query('search'));

        $query = AssetStatus::select('id', 'name')->orderBy('name');

        if (!empty($search)) {
            $query->where('name', 'LIKE', "%{$search}%");
        }

        $statuses = $query->limit($limit)->get();

        $data = [];
        foreach ($statuses as $status) {
            $data[] = [
                'id' => $status->id,
                'text' => ucfirst($status->name),
            ];
        }

        return $this->success(['statuses' => $data], 'Data Retrieved.');
    }
}
