<?php

namespace App\Http\Controllers\API\Setups;

use App\Http\Controllers\Controller;
use App\Models\Branch;
use App\Models\BranchPoint;
use App\Models\Traits\HttpResponses;
use Carbon\Carbon;
use Exception;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Gate;
use Illuminate\Support\Facades\Validator;
use Symfony\Component\HttpFoundation\Response;

class BranchPointApiController extends Controller
{
    use HttpResponses;

    public function index(Request $request)
    {
        if (!Gate::allows('View Branch Point')) {
            return $this->error('Permission Denied', [], Response::HTTP_FORBIDDEN);
        }

        $limit = $request->query->get('limit', 15);
        $search = $request->query->get('search');
        $branch_ids = $request->branch_ids;

        $query = BranchPoint::with('branch')->orderBy('name');

        if (!empty($search)) {
            $query->where('name', 'LIKE', "%{$search}%");
        }

        if (!empty($branch_ids)) {
            $query->whereIn('branch_id', $branch_ids);
        }

        $branch_points = $query->paginate($limit);

        $data = [];
        foreach ($branch_points as $branch_point) {
            $data[] = [
                'id' => $branch_point->id,
                'name' => $branch_point->name,
                'branch_name' => $branch_point->branch->name,
                'branch_id' => $branch_point->branch->id,
                'created_by' => $branch_point->createdBy->name,
                'status' => $branch_point->status,
                'status_label' => $branch_point->branch_point_status,
                'added_date' => Carbon::parse($branch_point->created_at)->format('d:m:Y h:i'),
            ];
        }

        return $this->success([
            'branch_points' => $data,
            'pagination' => [
                'total' => $branch_points->total(),
                'current_page' => $branch_points->currentPage(),
                'last_page' => $branch_points->lastPage(),
                'per_page' => $branch_points->perPage()
            ]
        ], 'Data Retrieved.', true);
    }

    public function show(Request $request)
    {
        if (!Gate::allows('View Branch Point')) {
            return $this->error('Permission Denied', [], Response::HTTP_FORBIDDEN);
        }

        $validate = Validator::make($request->all(), [
            'branch_point_id' => 'required|exists:branch_points,id',
        ]);

        if ($validate->fails()) {
            return $this->error('Validation Failed.', $validate->errors(), Response::HTTP_UNPROCESSABLE_ENTITY);
        }

        try {
            $branch_point = BranchPoint::with('branch')->find($request->branch_point_id);

            $data = [
                'id' => $branch_point->id,
                'name' => $branch_point->name,
                'branch_name' => $branch_point->branch->name,
                'branch_id' => $branch_point->branch->id,
                'created_by' => $branch_point->createdBy->name,
                'status' => $branch_point->status,
                'status_label' => $branch_point->branch_point_status,
            ];

            return $this->success(['branch_point' => $data], 'Data Retrieved.');
        } catch (Exception $e) {
            return $this->error('Error Occurred.', $e->getMessage(), Response::HTTP_UNPROCESSABLE_ENTITY);
        }
    }

    public function store(Request $request)
    {
        if (!Gate::allows('Create Branch Point')) {
            return $this->error('Permission Denied', [], Response::HTTP_FORBIDDEN);
        }

        $validate = Validator::make($request->all(), [
            'branch_id' => 'required|exists:branches,id',
            'name' => 'required|max:255',
        ]);

        if ($validate->fails()) {
            return $this->error('Validation Failed.', $validate->errors(), Response::HTTP_UNPROCESSABLE_ENTITY);
        }

        $check_branch_point = BranchPoint::where('name', $request->name)
            ->where('branch_id', $request->branch_id)
            ->first();

        if ($check_branch_point) {
            return $this->error([], 'Branch name is already exist.', Response::HTTP_UNPROCESSABLE_ENTITY);
        }

        try {
            BranchPoint::create([
                'branch_id' => $request->branch_id,
                'name' => $request->name,
                'status' => 1,
                'created_by' => Auth::id(),
            ]);

            return $this->success([], 'Data Created.', Response::HTTP_CREATED, true);
        } catch (Exception $e) {
            return $this->error('Error Occurred.', $e->getMessage(), Response::HTTP_UNPROCESSABLE_ENTITY);
        }
    }

    public function update(Request $request)
    {
        if (!Gate::allows('Update Branch Point')) {
            return $this->error('Permission Denied', [], Response::HTTP_FORBIDDEN);
        }

        $validate = Validator::make($request->all(), [
            'branch_point_id' => 'required|exists:branch_points,id',
            'branch_id' => 'required|exists:branches,id',
            'name' => 'required|max:255|unique:branch_points,name,' . $request->branch_point_id,
        ]);

        if ($validate->fails()) {
            return $this->error('Validation Failed.', $validate->errors(), Response::HTTP_UNPROCESSABLE_ENTITY);
        }

        try {
            $branch_point = BranchPoint::find($request->branch_point_id);
            $branch_point->update([
                'branch_id' => $request->branch_id,
                'name' => $request->name,
            ]);

            return $this->success([], 'Data Updated.', Response::HTTP_OK, true);
        } catch (Exception $e) {
            return $this->error('Error Occurred.', $e->getMessage(), Response::HTTP_UNPROCESSABLE_ENTITY);
        }
    }

    public function branchPointForSelect(Request $request)
    {
        $limit = $request->query('limit', 50);
        $search = trim($request->query('search'));

        $query = BranchPoint::select('id', 'name')->orderBy('name');

        if (!empty($search)) {
            $query->where('name', 'LIKE', "%{$search}%");
        }

        $branch_points = $query->limit($limit)->get();

        $data = [];
        foreach ($branch_points as $branch_point) {
            $data[] = [
                'id' => $branch_point->id,
                'text' => $branch_point->name,
            ];
        }

        return $this->success(['branch_points' => $data], 'Data Retrieved.');
    }

    public function checkBranchPoint(Request $request)
    {
        $validate = Validator::make($request->all(), [
            'branch_id' => 'required|exists:branches,id',
        ]);

        if ($validate->fails()) {
            return $this->error('Validation Failed.', $validate->errors(), Response::HTTP_UNPROCESSABLE_ENTITY);
        }

        try {
            $branch = Branch::with('branchPoints')->findOrFail($request->branch_id);

            $has_branch_point = $branch->has_branch_point;

            return $this->success([
                'branch_id' => $branch->id,
                'branch_name' => $branch->name,
                'has_points' => $has_branch_point,
                'points' => $has_branch_point ? $branch->branchPoints->map(function ($point) {
                    return [
                        'id' => $point->id,
                        'name' => $point->name,
                    ];
                }) : [],
            ], 'Data Retrieved.');
        } catch (Exception $e) {
            return $this->error('Error occurred.', $e->getMessage(), Response::HTTP_UNPROCESSABLE_ENTITY);
        }
    }
}
