<?php

namespace App\Http\Controllers\API\Setups;

use App\Http\Controllers\Controller;
use App\Models\Brand;
use App\Models\Traits\HttpResponses;
use Carbon\Carbon;
use Exception;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Gate;
use Illuminate\Support\Facades\Validator;
use Illuminate\Validation\Rule;
use Symfony\Component\HttpFoundation\Response;

class BrandApiController extends Controller
{
    use HttpResponses;

    public function index(Request $request)
    {
        if (!Gate::allows('View Brand')) {
            return $this->error('Permission Denied', [], Response::HTTP_FORBIDDEN);
        }

        $limit = $request->query->get('limit', 15);
        $search = $request->query->get('search');

        $query = Brand::latest();

        if (!empty($search)) {
            $query->where('name', 'LIKE', "%{$search}%");
        }

        $brands = $query->paginate($limit);

        $data = [];
        foreach ($brands as $brand) {
            $data[] = [
                'id' => $brand->id,
                'name' => $brand->name,
                'added_date' => Carbon::parse($brand->created_at)->format('d:m:Y h:i')
            ];
        }

        return $this->success([
            'brands' => $data,
            'pagination' => [
                'total' => $brands->total(),
                'current_page' => $brands->currentPage(),
                'last_page' => $brands->lastPage(),
                'per_page' => $brands->perPage()
            ]
        ], 'Data Retrieved.', true);
    }
    
    public function show(Request $request)
    {
        if (!Gate::allows('View Brand')) {
            return $this->error('Permission Denied', [], Response::HTTP_FORBIDDEN);
        }

        $validate = Validator::make($request->all(), [
            'brand_id' => 'required|exists:brands,id'
        ]);

        if ($validate->fails()) {
            return $this->error('Validation Failed.', $validate->errors(), Response::HTTP_UNPROCESSABLE_ENTITY);
        }

        try {
            $brand = Brand::findOrFail($request->brand_id);

            $data = [
                'id' => $brand->id,
                'name' => $brand->name,
                'descriptions' => $brand->descriptions ?? '',
            ];

            return $this->success([
                'brand' => $data,
            ], 'Data Retrieved.');
        } catch (Exception $e) {
            return $this->error('Error occurred.', $e->getMessage(), Response::HTTP_NOT_FOUND);
        }
    }


    public function store(Request $request)
    {
        if (!Gate::allows('Create Brand')) {
            return $this->error('Permission Denied', [], Response::HTTP_FORBIDDEN);
        }

        $validate = Validator::make($request->all(), [
            'name' => 'required|string|unique:brands,name',
            'descriptions' => 'nullable|string',
        ]);

        if ($validate->fails()) {
            return $this->error('Validation Failed.', $validate->errors(), Response::HTTP_UNPROCESSABLE_ENTITY);
        }

        try {
            Brand::create([
                'name' => $request->name,
                'descriptions' => $request->descriptions,
            ]);

            return $this->success([], 'Data created.', Response::HTTP_CREATED, true);
        } catch (Exception $e) {
            return $this->error('Error Occurred.', $e->getMessage(), Response::HTTP_UNPROCESSABLE_ENTITY);
        }
    }

    public function update(Request $request)
    {
        if (!Gate::allows('Update Brand')) {
            return $this->error('Permission Denied', [], Response::HTTP_FORBIDDEN);
        }

        $validate = Validator::make($request->all(), [
            'brand_id' => 'required|exists:brands,id',
            'name' => [
                'required',
                'string',
                Rule::unique('brands', 'name')->ignore($request->brand_id)
            ],
            'descriptions' => 'nullable|string|max:500',
        ]);

        if ($validate->fails()) {
            return $this->error('Validation Failed.', $validate->errors(), Response::HTTP_UNPROCESSABLE_ENTITY);
        }

        try {
            $brand = Brand::findOrFail($request->brand_id);

            $brand->update([
                'name' => $request->name,
                'descriptions' => $request->descriptions,
            ]);

            return $this->success([], 'Data updated.', Response::HTTP_OK, true);
        } catch (Exception $e) {
            return $this->error('Error Occurred.', $e->getMessage(), Response::HTTP_UNPROCESSABLE_ENTITY);
        }
    }

    public function brandForSelect(Request $request)
    {
        $limit = $request->query('limit', 50);
        $search = trim($request->query('search'));

        $query = Brand::select('id', 'name')->orderBy('name');

        if (!empty($search)) {
            $query->where('name', 'LIKE', "%{$search}%");
        }

        $brands = $query->limit($limit)->get();

        $data = [];
        foreach ($brands as $brand) {
            $data[] = [
                'id' => $brand->id,
                'text' => ucfirst($brand->name),
            ];
        }

        return $this->success(['brands' => $data], 'Data Retrieved.');
    }
}
