<?php

namespace App\Http\Controllers\API\Setups;

use App\Http\Controllers\Controller;
use App\Models\BrandModel;
use App\Models\Traits\HttpResponses;
use Exception;
use Illuminate\Http\Request;
use Illuminate\Support\Carbon;
use Illuminate\Support\Facades\Gate;
use Illuminate\Support\Facades\Validator;
use Illuminate\Validation\Rule;
use Symfony\Component\HttpFoundation\Response;

class BrandModelApiController extends Controller
{
    use HttpResponses;

    public function index(Request $request)
    {
        if (!Gate::allows('View Brand Model')) {
            return $this->error('Permission Denied', [], Response::HTTP_FORBIDDEN);
        }

        $limit = $request->query->get('limit', 15);
        $search = $request->query->get('search');

        $query = BrandModel::select('id', 'name', 'brand_id', 'status')
            ->latest();

        if (!empty($search)) {
            $query->where('name', 'LIKE', "%{$search}%")
                ->orWhereHas('brand', function ($sub_query) use ($search) {
                    $sub_query->where('name', 'LIKE', "%{$search}%");
                });
        }

        $brand_models = $query->paginate($limit);

        $data = [];
        foreach ($brand_models as $model) {
            $data[] = [
                'id' => $model->id,
                'name' => $model->name,
                'brand_name' => optional($model->brand)->name ?? '',
                'brand_id' => optional($model->brand)->id ?? '',
                'status' => $model->brand_model_status,
                'added_date' => Carbon::parse($model->created_at)->format('d:m:Y h:i')
            ];
        }

        return $this->success([
            'brand_models' => $data,
            'pagination' => [
                'total' => $brand_models->total(),
                'current_page' => $brand_models->currentPage(),
                'last_page' => $brand_models->lastPage(),
                'per_page' => $brand_models->perPage()
            ]
        ], 'Data Retrieved.', true);
    }

    public function show(Request $request)
    {
        if (!Gate::allows('View Brand Model')) {
            return $this->error('Permission Denied', [], Response::HTTP_FORBIDDEN);
        }

        $validate = Validator::make($request->all(), [
            'brand_model_id' => 'required|exists:brand_models,id',
        ]);

        if ($validate->fails()) {
            return $this->error('Validation Failed.', $validate->errors(), Response::HTTP_UNPROCESSABLE_ENTITY);
        }

        try {
            $model = BrandModel::with('brand:id,name')->findOrFail($request->brand_model_id);

            $data = [
                'id' => $model->id,
                'name' => $model->name,
                'brand_id' => $model->brand_id ?? '',
                'brand_name' => optional($model->brand)->name ?? '',
                'status' => $model->brand_model_status ?? ''
            ];

            return $this->success(['brand_model' => $data], 'Data Retrieved.');
        } catch (Exception $e) {
            return $this->error('Error occurred.', $e->getMessage(), Response::HTTP_UNPROCESSABLE_ENTITY);
        }
    }

    public function store(Request $request)
    {
        if (!Gate::allows('Create Brand Model')) {
            return $this->error('Permission Denied', [], Response::HTTP_FORBIDDEN);
        }

        $validate = Validator::make($request->all(), [
            'brand_id' => 'required|exists:brands,id',
            'name' => 'required|string|unique:brand_models,name',
        ]);

        if ($validate->fails()) {
            return $this->error('Validation Failed.', $validate->errors(), Response::HTTP_UNPROCESSABLE_ENTITY);
        }

        try {
            BrandModel::create([
                'brand_id' => $request->brand_id,
                'name' => $request->name,
            ]);

            return $this->success([], 'Data created.', Response::HTTP_CREATED, true);
        } catch (Exception $e) {
            return $this->error('Error Occurred.', $e->getMessage(), Response::HTTP_UNPROCESSABLE_ENTITY);
        }
    }

    public function update(Request $request)
    {
        if (!Gate::allows('Update Brand Model')) {
            return $this->error('Permission Denied', [], Response::HTTP_FORBIDDEN);
        }

        $validate = Validator::make($request->all(), [
            'brand_model_id' => 'required|exists:brand_models,id',
            'brand_id' => 'required|exists:brands,id',
            'name' => [
                'required',
                Rule::unique('brand_models', 'name')->ignore($request->brand_model_id)
            ],
        ]);

        if ($validate->fails()) {
            return $this->error('Validation Failed.', $validate->errors(), Response::HTTP_UNPROCESSABLE_ENTITY);
        }

        try {
            $model = BrandModel::findOrFail($request->brand_model_id);

            $model->update([
                'brand_id' => $request->brand_id,
                'name' => $request->name,
            ]);

            return $this->success([], 'Data updated.', Response::HTTP_OK, true);
        } catch (Exception $e) {
            return $this->error('Error Occurred.', $e->getMessage(), Response::HTTP_UNPROCESSABLE_ENTITY);
        }
    }

    public function brandModelForSelect(Request $request)
    {
        $limit = $request->query('limit', 50);
        $search = trim($request->query('search'));
        $brand_id = $request->query('brand_id');

        $query = BrandModel::select('id', 'name')->orderBy('name');

        if (!empty($brand_id)) {
            $query->where('brand_id', $brand_id);
        }

        if (!empty($search)) {
            $query->where('name', 'LIKE', "%{$search}%");
        }

        $brand_models = $query->limit($limit)->get();

        $data = [];
        foreach ($brand_models as $brand_model) {
            $data[] = [
                'id' => $brand_model->id,
                'text' => ucfirst($brand_model->name),
            ];
        }

        return $this->success(['brand_models' => $data], 'Data Retrieved.');
    }

    public function brandModelByBrandId($brand_id){
        return BrandModel::where('brand_id',$brand_id)->orderBy('name')->get(['id','name']);
    }
}
