<?php

namespace App\Http\Controllers\API\Setups;

use App\Http\Controllers\Controller;
use App\Models\Category;
use App\Models\Traits\HttpResponses;
use Exception;
use Illuminate\Http\Request;
use Illuminate\Support\Carbon;
use Illuminate\Support\Facades\Gate;
use Illuminate\Support\Facades\Validator;
use Illuminate\Validation\Rule;
use Symfony\Component\HttpFoundation\Response;

class CategoryApiController extends Controller
{
    use HttpResponses;

    public function index(Request $request)
    {
        if (!Gate::allows('View Category')) {
            $this->error('Permission Denied', [], Response::HTTP_FORBIDDEN);
        }

        $limit = $request->query->get('limit', 15);
        $search = $request->query->get('search');
        $category_group_ids = $request->category_group_ids;

        $query = Category::with('categoryGroup:id,name')->latest();

        if (!empty($search)) {
            $query->where('name', 'LIKE', "%{$search}%")
                ->orWhereHas('categoryGroup', function ($sub_query) use ($search) {
                    $sub_query->where('name', 'LIKE', "%{$search}%");
                });
        }

        if (!empty($category_group_ids)) {
            $validate = Validator::make($request->all(), [
                'category_group_ids' => 'array',
            ]);

            if ($validate->fails()) {
                return $this->error('Validation Failed.', $validate->errors(), Response::HTTP_UNPROCESSABLE_ENTITY);
            }

            $query->whereIn('category_group_id', $category_group_ids);
        }

        $categories = $query->paginate($limit);

        $data = [];
        foreach ($categories as $category) {
            $data[] = [
                'id' => $category->id,
                'name' => $category->name,
                'category_group_name' => optional($category->categoryGroup)->name ?? '',
                'category_group_id' => optional($category->categoryGroup)->id ?? '',
                'expected_useful_life' => $category->expected_useful_life ?? '',
                'added_date' => Carbon::parse($category->created_at)->format('d:m:Y h:i')
            ];
        }

        return $this->success([
            'categories' => $data,
            'pagination' => [
                'total' => $categories->total(),
                'current_page' => $categories->currentPage(),
                'last_page' => $categories->lastPage(),
                'per_page' => $categories->perPage()
            ]
        ], 'Data Retrieved.', true);
    }

    public function show(Request $request)
    {
        if (!Gate::allows('View Category')) {
            return $this->error('Permission Denied', [], Response::HTTP_FORBIDDEN);
        }

        $validate = Validator::make($request->all(), [
            'category_id' => 'required|exists:categories,id',
        ]);

        if ($validate->fails()) {
            return $this->error('Validation Failed.', $validate->errors(), Response::HTTP_UNPROCESSABLE_ENTITY);
        }

        try {
            $category = Category::with('categoryGroup:id,name')->findOrFail($request->category_id);
            $data =  [
                'id' => $category->id,
                'name' => $category->name,
                'category_group_id' => $category->category_group_id ?? '',
                'category_group_name' => optional($category->categoryGroup)->name ?? '',
            ];

            return $this->success([
                'category' => $data,
            ], 'Data Retrieved.');
        } catch (Exception $e) {
            return $this->error('Error occurred.', $e->getMessage(), Response::HTTP_UNPROCESSABLE_ENTITY);
        }
    }

    public function store(Request $request)
    {
        if (!Gate::allows('Create Category')) {
            $this->error('Permission Denied', [], Response::HTTP_FORBIDDEN);
        }

        $validate = Validator::make($request->all(), [
            'name' => 'required|unique:categories,name',
            'category_group_id' => 'required|exists:category_groups,id',
            'expected_useful_life' => 'nullable'
        ]);

        if ($validate->fails()) {
            return $this->error('Validation Failed.', $validate->errors(), Response::HTTP_UNPROCESSABLE_ENTITY);
        }

        try {
            Category::create([
                'name' => $request->name,
                'category_group_id' => $request->category_group_id,
                'expected_useful_life' => $request->expected_useful_life
            ]);

            return $this->success([], 'Data created.', Response::HTTP_CREATED, true);
        } catch (Exception $e) {
            return $this->error('Error Occurred.', $e->getMessage(), Response::HTTP_UNPROCESSABLE_ENTITY);
        }
    }

    public function update(Request $request)
    {
        if (!Gate::allows('Update Category')) {
            $this->error('Permission Denied', [], Response::HTTP_FORBIDDEN);
        }

        $validate = Validator::make($request->all(), [
            'category_id' => 'required|exists:categories,id',
            'name' => [
                'required',
                Rule::unique('categories', 'name')->ignore($request->category_id)
            ],
            'category_group_id' => 'required|exists:category_groups,id',
            'expected_useful_life' => 'nullable'
        ]);

        if ($validate->fails()) {
            return $this->error('Validation Failed.', $validate->errors(), Response::HTTP_UNPROCESSABLE_ENTITY);
        }

        try {
            $category = Category::findOrFail($request->category_id);

            $category->update([
                'name' => $request->name,
                'category_group_id' => $request->category_group_id,
                'expected_useful_life' => $request->expected_useful_life
            ]);

            return $this->success([], 'Data updated.', Response::HTTP_OK, true);
        } catch (Exception $e) {
            return $this->error('Error Occurred.', $e->getMessage(), Response::HTTP_UNPROCESSABLE_ENTITY);
        }
    }

    public function categoryForSelect(Request $request)
    {
        $limit = $request->query('limit', 50);
        $search = trim($request->query('search'));
        $category_group_id = $request->query('category_group_id');

        $query = Category::select('id', 'name')
            ->orderBy('name');

        if (!empty($category_group_id)) {
            $query->where('category_group_id', $category_group_id);
        };

        if (!empty($search)) {
            $query->where('name', 'LIKE', "%{$search}%");
        }

        $categories = $query->limit($limit)->get();

        $data = [];
        foreach ($categories as $category) {
            $data[] = [
                'id' => $category->id,
                'text' => ucfirst($category->name),
            ];
        }

        return $this->success(['categories' => $data], 'Data Retrieved.');
    }
}
