<?php

namespace App\Http\Controllers\API\Setups;

use App\Http\Controllers\Controller;
use App\Models\CategoryGroup;
use App\Models\Traits\HttpResponses;
use Carbon\Carbon;
use Exception;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Gate;
use Illuminate\Support\Facades\Validator;
use Symfony\Component\HttpFoundation\Response;

class CategoryGroupApiController extends Controller
{
    use HttpResponses;

    /**
     * @OA\Get(
     *     path="/api/v1/setups/category-groups/index",
     *     summary="Get all category groups",
     *     tags={"Category Groups"},
     *     security={{"bearerAuth": {}}},
     *     @OA\Response(
     *         response=200,
     *         description="Successful retrieval"
     *     ),
     *     @OA\Response(
     *         response=403,
     *         description="Unauthorized"
     *     )
     * )
     */

    public function index(Request $request)
    {
        if (!Gate::allows('View Category Group')) {
            return $this->error('Permission Denied.', [], Response::HTTP_FORBIDDEN);
        }

        $limit = $request->query->get('limit', 15);
        $search = $request->query->get('search');

        $query = CategoryGroup::select('id', 'name', 'is_assignable', 'created_at')->latest();

        if (!empty($search)) {
            $query->where('name', 'LIKE', "%{$search}%");
        }

        $category_groups = $query->paginate($limit);

        $data = [];
        foreach ($category_groups as $category_group) {
            $data[] = [
                'id' => $category_group->id,
                'name' => $category_group->name,
                'is_assignable' => $category_group->is_assignable_badge,
                'added_date' => Carbon::parse($category_group->created_at)->format('d:m:Y h:i')
            ];
        }

        return $this->success([
            'category_groups' => $data,
            'pagination' => [
                'total' => $category_groups->total(),
                'current_page' => $category_groups->currentPage(),
                'last_page' => $category_groups->lastPage(),
                'per_page' => $category_groups->perPage()
            ]
        ], 'Data Retrieved.', true);
    }

    public function show(Request $request)
    {
        if (!Gate::allows('View Category Group')) {
            return $this->error('Permission Denied.', [], Response::HTTP_FORBIDDEN);
        }

        $validate = Validator::make($request->all(), [
            'category_group_id' => 'required|exists:category_groups,id',
        ]);

        if ($validate->fails()) {
            return $this->error('Validation Failed.', $validate->errors(), Response::HTTP_UNPROCESSABLE_ENTITY);
        }

        try {
            $category_group = CategoryGroup::find($request->category_group_id);

            if (!$category_group) {
                return $this->error('Data not found.', [], Response::HTTP_NOT_FOUND);
            }

            $data = [
                'id' => $category_group->id,
                'name' => $category_group->name,
                'descriptions' => $category_group->descriptions ?? '',
                'is_assignable' => $category_group->is_assignable_badge
            ];

            return $this->success(['category_group' => $data], 'Data Retrieved');
        } catch (Exception $e) {
            return $this->error('Error Occurred.', $e->getMessage(), Response::HTTP_UNPROCESSABLE_ENTITY);
        }
    }

    public function store(Request $request)
    {
        if (!Gate::allows('Create Category Group')) {
            return $this->error('Permission Denied.', [], Response::HTTP_FORBIDDEN);
        }
        $validate = Validator::make($request->all(), [
            'name' => 'required|string|unique:category_groups,name',
            'is_assignable' => 'required|in:1,0',
            'descriptions' => 'nullable|string'
        ]);

        if ($validate->fails()) {
            return $this->error('Validation Failed.', $validate->errors(), Response::HTTP_UNPROCESSABLE_ENTITY);
        }

        try {
            CategoryGroup::create([
                'name' => $request->name,
                'descriptions' => $request->descriptions,
                'is_assignable' => $request->is_assignable
            ]);

            return $this->success([], 'Data created.', Response::HTTP_CREATED, true);
        } catch (Exception $e) {
            return $this->error('Error Occurred.', $e->getMessage(), Response::HTTP_UNPROCESSABLE_ENTITY);
        }
    }

    public function update(Request $request)
    {
        if (!Gate::allows('Update Category Group')) {
            return $this->error('Permission Denied.', [], Response::HTTP_FORBIDDEN);
        }
        $validate = Validator::make($request->all(), [
            'category_group_id' => 'required|exists:category_groups,id',
            'name' => 'required|string',
            'is_assignable' => 'required|in:1,0',
            'descriptions' => 'nullable|string',
        ]);

        if ($validate->fails()) {
            return $this->error('Validation Failed.', $validate->errors(), Response::HTTP_UNPROCESSABLE_ENTITY);
        }

        try {
            $category_group = CategoryGroup::findOrFail($request->category_group_id);

            $category_group->update([
                'name' => $request->name,
                'descriptions' => $request->descriptions,
                'is_assignable' => $request->is_assignable
            ]);

            return $this->success([], 'Data created.', Response::HTTP_CREATED, true);
        } catch (Exception $e) {
            return  $this->error('Error Occurred.', $e->getMessage(), Response::HTTP_UNPROCESSABLE_ENTITY);
        }
    }

    public function categoryGroupForSelect(Request $request)
    {
        $limit = $request->query('limit', 50);
        $search = trim($request->query('search'));

        $query = CategoryGroup::select('id', 'name')->orderBy('name');

        if (!empty($search = trim($search))) {
            $query->where('name', 'LIKE', "%{$search}%");
        }

        $category_groups = $query->limit($limit)->get();

        $data = [];
        foreach ($category_groups as $category_group) {
            $data[] = [
                'id' => $category_group->id,
                'text' => ucfirst($category_group->name),
            ];
        }

        return $this->success(['category_groups' => $data], 'Data Retrieved.');
    }
}
