<?php

namespace App\Http\Controllers\API\Setups;

use App\Http\Controllers\Controller;
use App\Models\Traits\HttpResponses;
use App\Models\Vendor;
use Carbon\Carbon;
use Exception;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Gate;
use Illuminate\Support\Facades\Validator;
use Illuminate\Validation\Rule;
use Symfony\Component\HttpFoundation\Response;

class VendorApiController extends Controller
{
    use HttpResponses;

    public function index(Request $request)
    {
        if (!Gate::allows('View Vendor')) {
            return $this->error('Permission Denied', [], Response::HTTP_FORBIDDEN);
        }

        $limit = $request->query->get('limit', 15);
        $search = $request->search;

        $query = Vendor::latest();

        if (!empty($search)) {
            $query->where('name', 'LIKE', "%{$search}%");
        }

        $vendors = $query->paginate($limit);
        $data = [];
        foreach ($vendors as $vendor) {
            $data[] = [
                'id' => $vendor->id,
                'name' => $vendor->name,
                'email' => $vendor->email ?? '',
                'phone' => $vendor->phone ?? '',
                'contact_person' => $vendor->contact_person,
                'address' => $vendor->address ?? '',
                'added_date' => Carbon::parse($vendor->created_at)->format('d:m:Y h:i'),
            ];
        }

        return $this->success([
            'vendors' => $data,
            'pagination' => [
                'total' => $vendors->total(),
                'current_page' => $vendors->currentPage(),
                'last_page' => $vendors->lastPage(),
                'per_page' => $vendors->perPage()
            ]
        ], 'Data Retrieved.', true);
    }

    public function show(Request $request)
    {
        if (!Gate::allows('View Vendor')) {
            return $this->error('Permission Denied', [], Response::HTTP_FORBIDDEN);
        }

        $validate = Validator::make($request->all(), [
            'vendor_id' => 'required|exists:vendors,id',
        ]);

        if ($validate->fails()) {
            return $this->error('Validation Failed.', $validate->errors(), Response::HTTP_UNPROCESSABLE_ENTITY);
        }

        try {
            $vendor = Vendor::findOrFail($request->vendor_id);

            $data = [
                'id' => $vendor->id,
                'name' => $vendor->name,
                'email' => $vendor->email ?? '',
                'phone' => $vendor->phone ?? '',
                'contact_person' => $vendor->contact_person,
                'address' => $vendor->address ?? '',
            ];

            return $this->success(['vendor' => $data], 'Data Retrieved.', true);
        } catch (Exception $e) {
            return $this->error('Error occurred.', $e->getMessage(), Response::HTTP_UNPROCESSABLE_ENTITY);
        }
    }

    public function store(Request $request)
    {
        if (!Gate::allows('Create Vendor')) {
            return $this->error('Permission Denied', [], Response::HTTP_FORBIDDEN);
        }

        $validate = Validator::make($request->all(), [
            'name' => 'required|string|unique:vendors,name',
            'email' => 'nullable|email',
            'phone' => 'required|string',
            'contact_person' => 'required|string',
            'address' => 'required|string',
        ]);

        if ($validate->fails()) {
            return $this->error('Validation Failed.', $validate->errors(), Response::HTTP_UNPROCESSABLE_ENTITY);
        }

        try {
            Vendor::create([
                'name' => $request->name,
                'email' => $request->email,
                'phone' => $request->phone,
                'contact_person' => $request->contact_person,
                'address' => $request->address
            ]);

            return $this->success([], 'Data created.', Response::HTTP_CREATED, true);
        } catch (Exception $e) {
            return $this->error('Error Occurred.', $e->getMessage(), Response::HTTP_UNPROCESSABLE_ENTITY);
        }
    }

    public function update(Request $request)
    {
        if (!Gate::allows('Update Vendor')) {
            return $this->error('Permission Denied', [], Response::HTTP_FORBIDDEN);
        }

        $validate = Validator::make($request->all(), [
            'vendor_id' => 'required|exists:vendors,id',
            'name' => [
                'required',
                Rule::unique('vendors', 'name')->ignore($request->vendor_id)
            ],
            'email' => 'nullable|email',
            'phone' => 'nullable|string',
            'contact_person' => 'nullable|string',
            'address' => 'nullable|string',
        ]);

        if ($validate->fails()) {
            return $this->error('Validation Failed.', $validate->errors(), Response::HTTP_UNPROCESSABLE_ENTITY);
        }

        try {
            $vendor = Vendor::findOrFail($request->vendor_id);
            $vendor->update($request->only(['name', 'email', 'phone', 'contact_person', 'address']));

            return $this->success([], 'Data updated.', Response::HTTP_OK, true);
        } catch (Exception $e) {
            return $this->error('Error Occurred.', $e->getMessage(), Response::HTTP_UNPROCESSABLE_ENTITY);
        }
    }

    public function vendorForSelect(Request $request)
    {
        $limit = $request->query('limit', 50);
        $search = $request->query('search');

        $query = Vendor::select('id', 'name')->orderBy('name');

        if (!empty($search = trim($search))) {
            $query->where('name', 'LIKE', "%{$search}%");
        }

        $vendors = $query->limit($limit)->get();

        $data = [];
        foreach ($vendors as $vendor) {
            $data[] = [
                'id' => $vendor->id,
                'text' => ucfirst($vendor->name),
            ];
        }

        return $this->success(['vendors' => $data], 'Data Retrieved.');
    }
}
