<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;

class Asset extends Model
{
    protected $fillable = [
        'id',
        'name',
        'category_id',
        'vendor_id',
        'branch_id',
        'status_id',
        'condition_id',
        'brand_model_id',
        'department_id',
        'serial_number',
        'model_number',
        'asset_tag',
        'capitalization_date',
        'acquisition_date',
        'descriptions',
        'created_by',
        //other parameters
        'building',
        'room',
        'floor',
        'old_status',
    ];

    public function category()
    {
        return $this->belongsTo(Category::class);
    }

    public function user()
    {
        return $this->belongsTo(User::class, 'created_by');
    }

    public function brandModel()
    {
        return $this->belongsTo(BrandModel::class, 'brand_model_id');
    }

    public function vendor()
    {
        return $this->belongsTo(Vendor::class);
    }

    public function assetStatus()
    {
        return $this->belongsTo(AssetStatus::class, 'status_id');
    }

    public function assetCondition()
    {
        return $this->belongsTo(AssetCondition::class, 'condition_id');
    }

    public function department()
    {
        return $this->belongsTo(Department::class);
    }

    public function branch()
    {
        return $this->belongsTo(Branch::class);
    }

    public function createdBy()
    {
        return $this->belongsTo(User::class, 'created_by');
    }

    public function getAssetNameAttribute()
    {
        return $this->category->group_name;
    }

    public function getAssetWithNumberAttribute()
    {
        return strtoupper($this->category?->name . ' (' . $this->serial_number . ')');
    }

    public function getIsAllocatedAttribute()
    {
        $asset_allocation = $this->assetAllocations->where('status', 1)->first();
        return $asset_allocation ? 1 : 0;
    }

    public function getAllocationStatusAttribute()
    {
        return $this->is_allocated ? 'Allocated' : 'Not Allocated';
    }

    public function assetStateHistories()
    {
        return $this->hasMany(AssetStateHistory::class);
    }

    public function assetAllocations()
    {
        return $this->hasMany(AssetAllocation::class);
    }

    public function getCurrentAllocationAttribute()
    {
        return $this->assetAllocations
            ->where('status', 1)
            ->first();
    }

    public function getIsEmployeeAssignedAttribute()
    {
        return $this->current_allocation && $this->current_allocation->employee_id ? 1 : 0;
    }

    public function getIsBranchAssignedAttribute()
    {
        return $this->current_allocation && is_null($this->current_allocation->employee_id) ? 1 : 0;
    }

    public function getAllocatedEntityAttribute()
    {
        $allocation = $this->current_allocation;

        if (!$allocation) {
            return 'Not Assigned';
        }

        if ($allocation->employee_id && $allocation->employee) {
            return $allocation->employee->name;
        }

        if (is_null($allocation->employee_id) && $allocation->branch) {
            return $allocation->branch->name;
        }

        return 'Not Assigned';
    }

    public function assetMaintenances()
    {
        return $this->hasMany(AssetMaintenance::class);
    }

    public function getModelNumberAttribute()
    {
        return isset($this->attributes['model_number'])
            ? ucwords(strtolower($this->attributes['model_number']))
            : null;
    }
}
