<?php

namespace App\Models\Traits;

use Carbon\Exceptions\Exception;
use Illuminate\Http\Response;
use Illuminate\Support\Facades\Auth;
use PHPOpenSourceSaver\JWTAuth\Exceptions\JWTException;
use PHPOpenSourceSaver\JWTAuth\Facades\JWTAuth;
use Symfony\Component\HttpFoundation\Response as ResponseAlias;

trait HttpResponses
{
    // responses
    protected function respond($data = null, $message = null, $code = 200, $token = null)
    {
        $response = [
            'message' => $message,
            'data'    => $data,
        ];

        if ($token) {
            $response['token'] = $token;
        }

        return response()->json($response, $code);
    }

    //success response
    protected function success($data = null, $message = 'Success', $code = 200, $with_token = false)
    {
        if (is_bool($code)) {
            $with_token = $code;
            $code = 200;
        }

        $token = $with_token ? $this->refreshToken() : null;

        return $this->respond($data, $message, $code, $token);
    }


    //success error
    protected function error($message = 'Error', $errors = [], $code = 400)
    {
        return response()->json([
            'message' => $message,
            'errors'  => $errors,
        ], $code);
    }

    protected function refreshToken()
    {
        try {
            if (!$token = JWTAuth::getToken()) {
                return null;
            }

            $user = JWTAuth::authenticate($token);
            Auth::logout();

            return JWTAuth::fromUser($user);
        } catch (Exception $e) {
            return null;
        }
    }

    // usage
    // return $this->success($user, 'User data fetched');
    // return $this->success($user, 'Login successful', 200, true);
    // return $this->error('Invalid credentials', [], 401);

    protected static function recordFailed($message)
    {
        return response()->json(
            ['error' => 'Something went wrong while creating the record. Please try again later.', 'message' => $message],
            ResponseAlias::HTTP_INTERNAL_SERVER_ERROR
        )->setStatusCode(ResponseAlias::HTTP_INTERNAL_SERVER_ERROR, Response::$statusTexts[ResponseAlias::HTTP_INTERNAL_SERVER_ERROR]);
    }

    protected static function recordedCreated($data)
    {
        return response()->json([
            'data' => $data,
            'message' => 'Record Created successful',
            'status' => ResponseAlias::HTTP_CREATED
        ], ResponseAlias::HTTP_CREATED);
    }
}
